#define SUN_RISE_SET
/*
   wtravel.c
   World Travel information
   Jim Gasbarro
   Feb, 1996
*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <dir.h>

#include "wdat.h"
#include "pal.h"
#include "aljuli.h"
#include "sunriset.h"
#include "vec_math.c"

#define FONT_HT         10
#define TAB_KEY         9
#define MAX_WAY_PT      100
/*#define STR_LEN         64*/
#define STR_LEN         80
#define LF              '\015'
#define BELL            '\007'

#define CITYS  0
#define WAYPTS 1
#define DELTA  2
#define DEPART 3
#define ARRIVE 4

#define MAP_NAME 0

#define MI_PER_DEG  69.04674
#define KM_PER_DEG 111.11957
#define MI_PER_KM    0.62137119
#define KM_PER_MI    1.60934400
#define DEG2RAD      0.01745329252
#define RAD2DEG     57.2957795131
#define PI           3.1415926535

#define SUP_KEY  0x4838
#define SDN_KEY  0x5032
#define SLT_KEY  0x4B34
#define SRT_KEY  0x4D36
#define FCUP_KEY 0x8400
#define FCDN_KEY 0x7600
#define FCLT_KEY 0x7700
#define FCRT_KEY 0x7500

#define BANNER " Jim Gasbarro/Alfred Lee        World Travel V2.0"

extern CITY cities[];

typedef struct {
   int   City;
   float   Dist;
   float   Delta;
} WAY_PT;

FILE *fp;

WAY_PT WayPts[MAX_WAY_PT];
char blink_on = 0;

int bombmode;
int t_marg;
int b_marg;
int l_marg;
int r_marg;
float mid_lat;
float mid_lon;
float xspan;
float yspan;

int oldFI = -1;
int oldTI = -1;
int oldWI = -1;
int oldMiFlag = 1;
int WayCnt = 0;

int CHx = 320;
int CHy = 100;
int FirstCH = 1;

char cur_map[STR_LEN];
char credit[] = "Portions copyright 1995, The PAL Group";




#define CITY_T 0
#define COUNTRY_T 1
#define CITYNAME 2
#define COUNTRY 3
#define LONG_T 4
#define LONG 5
#define LAT 6
#define LAT_T 7
#define RISE_LABEL 8
#define SUN_T 9
#define SUN_RISE 10
#define SUN_SET  11
#define SUN_LEN  12
#define CIVIL_T 13
#define CIVIL_RISE 14
#define CIVIL_SET  15
#define CIVIL_LEN  16
#define ALT_LABEL 17
#define POSITION_T 18
#define POSITION 19
#define DATETIME 20
#define NOTE 21
#define BT_CLOSE 22
#define BT_ADD_YEAR  23
#define BT_SUB_YEAR  24
#define BT_ADD_MONTH 25
#define BT_SUB_MONTH 26
#define BT_ADD_WEEK  27
#define BT_SUB_WEEK  28
#define BT_ADD_DAY   29
#define BT_SUB_DAY   30
#define BT_ADD_HOUR  31
#define BT_SUB_HOUR  32
#define BT_ADD_MIN   33
#define BT_SUB_MIN   34
#define BT_RISET     35
#define BT_NOW       36


DLGITEM sunItems[] = {
/* X    Y    W   D   FLAGS     LABEL               INIT       TYPE           SIZE        PRIV */

{ 10,   5,  90, 10,   IF_STLFT, "City    :",             NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{ 10,  19,  90, 10,   IF_STLFT, "Country :",             NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{120,   5, 200, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{120,  19, 200, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{ 10,  33,  90, 10,   IF_STLFT, "Long    :",             NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{120,  33,  80, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{290,  33,  80, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{230,  33,  50, 10,   IF_STLFT, "Lat :",                 NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{130,  51, 210, 10,   IF_STLFT, "Rise   Set  Day len",   NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{ 10,  65,  90, 10,   IF_STLFT, "Sun     :",             NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{120,  65, 200, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{190,  65,  60, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{260,  65,  60, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{ 10,  79,  90, 10,   IF_STLFT, "Civil   :",             NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{120,  79, 200, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{190,  79,  60, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{260,  79,  60, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{220,  97,  80, 10,   IF_STLFT, "Azimuth  Altitude",     NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{ 10, 111, 200, 10,   IF_STLFT, "Current Position :",    NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{210, 111, 200, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{140, 129, 300, 10,   IF_STLFT, "",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },
{ 10, 143, 430, 10,   IF_STLFT, "alfred1520@aol.com for sun calc questions",                      NULL,       IhStatic, sizeof(IHSTATIC), NULL },

{460, 145,  70, 16,         0L, "Close",                 NULL,       IhButton, sizeof(IHBUTTON), NULL },
{440,   5,  40, 16,         0L, "+y",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{490,   5,  40, 16,         0L, "-y",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{440,  25,  40, 16,         0L, "+m",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{490,  25,  40, 16,         0L, "-m",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{440,  45,  40, 16,         0L, "+w",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{490,  45,  40, 16,         0L, "-w",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{440,  65,  40, 16,         0L, "+d",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{490,  65,  40, 16,         0L, "-d",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{440,  85,  40, 16,         0L, "+h",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{490,  85,  40, 16,         0L, "-h",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{440, 105,  40, 16,         0L, "+m",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{490, 105,  40, 16,         0L, "-m",                    NULL,       IhButton, sizeof(IHBUTTON), NULL },
{440, 125,  40, 16,         0L, "R/S",                   NULL,       IhButton, sizeof(IHBUTTON), NULL },
{490, 125,  40, 16,         0L, "Now",                   NULL,       IhButton, sizeof(IHBUTTON), NULL },

{ 10, 129, 300, 10,   IF_STLFT, "Local Time :",          NULL,       IhStatic, sizeof(IHSTATIC), NULL },

};

DIALOG sundlg = {
545, 180,
0L,
0,
38,
sunItems,
DhStandard,
NULL,
NULL,
0
};




DLGITEM TravelDlgItems[] = {
 {  3, 60, 180, 114, 0L, "&Cities", NULL, IhListBox, sizeof(IHLISTBOX), NULL},
 {190, 60, 180, 114, 0L, "&Way Points", NULL, IhListBox, sizeof(IHLISTBOX), NULL},
 {390, 112, -5, 14, 0L, "Way &Pt Delta: ", NULL, IhEdit, sizeof(IHEDIT), NULL},
};

DIALOG TravelDlg = {
   640, 200, 0L, 0, 3, TravelDlgItems, DhStandard, NULL, NULL, 0
};

DLGITEM MapNameItems[] = {
  {5, 10, -10, 14, 0L, "Map Name: ", NULL, IhEdit, sizeof(IHEDIT), NULL},
};

DIALOG MapNameDlg = {
  220, 40, 0L, 0, 1, MapNameItems, DhStandard, NULL, NULL, 0
};

char *ListFnKeys[] = { /*alal*/
  "From","To/Add","Mi","WayPts","ClrTot","AddTot","Sun","Map","SCtry","Quit"
};

char *MapFnKeys[] = {
  "From","To/Add","Route","WayPts","ClrMap","LdMap","NearBy","List","SCtry","Quit"
};

void alsun (int CurCity, int WIndex, int Focus, char *Str);


char *al_f_hm (double f, char h, char m) {
    static char buf [12];
    int i, j, sign;

    if (f < 0.) {
        sign = -1;
        f = -f;
    } else {
        sign = 1;
    }

    i = (int) f;

    j = (int) ((f - i) * 60);

    if (sign < 0) {
        if (i == 0) {
            sprintf (buf, "  -0%c%02d%c", h, j, m);
        } else {
            sprintf (buf, "%4d%c%02d%c", -i, h, j, m);
        }
    } else {
        sprintf (buf, "%4d%c%02d%c", i, h, j, m);
    }

    return (buf);

} /* al_f_hm */


void ClearStr(char *str)
{
  *str = '\0';
}

void Text(int x, int y, int w, char *str)
{
   SetRule(FORCE_RULE);
   SetColor(WHITE_COLOR);
   Rectangle(x, y, x+w, y+FONT_HT, SOLID_FILL);
   SetColor(BLACK_COLOR);
   SetClip(x, y, x+w, y+FONT_HT);
   WriteText(x, y, str);
   SetClip(0, 0, 640, 200);
}

float Dist(int City1, int City2, int MiFlag)
{
  float ang;
  double dlat1, dlon1, dlat2, dlon2;

  dlat1 = cities[City1].n_lat * DEG2RAD;
  dlon1 = cities[City1].w_lon * DEG2RAD;
  dlat2 = cities[City2].n_lat * DEG2RAD;
  dlon2 = cities[City2].w_lon * DEG2RAD;

  /* calculate great circle angle */
  ang = acos(sin(dlat1)*sin(dlat2) + cos(dlat1)*cos(dlat2)*cos(dlon1-dlon2));
  return((ang/DEG2RAD) * (MiFlag ? MI_PER_DEG : KM_PER_DEG));
}

char *LbCbCity(int n)
{
   return (cities[n].name);
}

char *LbCbWayPt(int n)
{
   if (WayCnt == 0) return ("");
   else return (cities[WayPts[n].City].name);
}

void InitItems(int nCitys, float Delta)
{
   char str[STR_LEN];
   LbSetCallBack(&TravelDlg, CITYS, LbCbCity);
   LbSetCount(&TravelDlg, CITYS, nCitys);
   LbSetCallBack(&TravelDlg, WAYPTS, LbCbWayPt);
   LbSetCount(&TravelDlg, WAYPTS, (WayCnt == 0) ? 1 : WayCnt);
   sprintf(str, "%.0f", Delta);
   SetDlgItem(&TravelDlg, DELTA, EDSI_TXT, str);
}

int AppendChar(char *str, int c)
{
   int length;

   length = strlen(str);
   switch (c) {
      case TAB_KEY:
      case STAB_KEY:
         break;
      case BACK_KEY:
         if (length > 0) str[--length] = '\0';
         break;
      case ESC_KEY:
         str[0] = '\0';
         length = 0;
         break;
      default:
         if (length < STR_LEN) {
            str[length++] = c;
            str[length] = '\0';
         } else putchar('\007');
         break;
   }
   return length;
}

void CitySearch(int *Index, char *SearchStr, int Key, int CityCnt,
     int SearchCities)
{
   int length, i, j, StartIndex;
   static int LastIndex = 0;

   if (Key == '+') Key = TAB_KEY;
   if (Key == '-') Key = STAB_KEY;
   if ( ((Key < 'a') || (Key > 'z')) &&
        ((Key < 'A') || (Key > 'Z')) &&
        (Key != ' ') && (Key != '.') &&
        (Key != ESC_KEY) &&
        (Key != BACK_KEY) &&
        (Key != TAB_KEY) &&
        (Key != STAB_KEY)
      ) return;

   if ((Key != TAB_KEY) && (Key != STAB_KEY)) StartIndex = 0;
   else StartIndex = (Key == TAB_KEY) ? ++LastIndex : --LastIndex;
   length = AppendChar(SearchStr, Key);

   for (i=StartIndex; (Key == STAB_KEY) ? i>=0 : i<CityCnt;
    (Key == STAB_KEY) ? i-- : i++) {
      for (j=0; j<length; j++)
         if (SearchCities) {
           if ((cities[i].name[j] & 0x5f) != (SearchStr[j] & 0x5f)) break;
         }
         else if ((cities[cities[i].country].cname[j] & 0x5f) !=
                 (SearchStr[j] & 0x5f)) break;

      if (j == length) {
         *Index = i;
         LastIndex = i;
         break;
      }
   }
   if (i == CityCnt) {
      if (Key == TAB_KEY) LastIndex--;
      else SearchStr[--length] = '\0';
      putchar('\007');
   }
   if ((i==-1) && (Key == STAB_KEY)) {
     LastIndex++;
     putchar('\007');
   }
}

void SortWayPts()
{
   int i, j, end, max;
   WAY_PT w;

   end = WayCnt;
   for(j=0; j<WayCnt; j++) {
      w.Dist = -1;
      for (i=0; i<end; i++) {
         if (WayPts[i].Dist > w.Dist) {
            w.Dist = WayPts[i].Dist;
            w.City = WayPts[i].City;
            w.Delta = WayPts[i].Delta;
            max = i;
         }
      }
      WayPts[max].Dist = WayPts[end-1].Dist;
      WayPts[max].City = WayPts[end-1].City;
      WayPts[max].Delta = WayPts[end-1].Delta;
      WayPts[end-1].Dist = w.Dist;
      WayPts[end-1].City = w.City;
      WayPts[end-1].Delta = w.Delta;
      --end;
   }
}


void FindWayPt(int FIndex, int TIndex, int CityCnt, float Delta, int MiFlag)
{
   float distance, d1, d2, lat, lat1, lat2, lon, lon1, lon2, p, DDelta;
   int i, GT180;

   WayCnt = 0;

   /* convert Delta to degrees */
   DDelta = Delta/(MiFlag ? MI_PER_DEG : KM_PER_DEG);

   lat1 = cities[FIndex].n_lat;
   lat2 = cities[TIndex].n_lat;
   lon1 = cities[FIndex].w_lon;
   lon2 = cities[TIndex].w_lon;

   /* make lat1 the southern-most city */
   if (lat1 > lat2) {
      lat = lat1;
      lat1 = lat2;
      lat2 = lat;
   }
   /* add and subtract Delta radius to endpoints */
   lat1 -= DDelta;
   lat2 += DDelta;

   /* make lon1 the eastern-most city */
   if (lon1 > lon2) {
      lon = lon1;
      lon1 = lon2;
      lon2 = lon;
   }
   /* add or subtract Delta radius to endpoints */
   if ((lon2 - lon1) < 180) {lon1 -= DDelta; lon2 += DDelta; GT180 = 0;}
   else {lon1 += DDelta; lon2 -= DDelta; GT180 = 1;}

   if ((FIndex < 0) || (TIndex < 0)) return;
   distance = Dist(FIndex, TIndex, MiFlag);
   /* find the way point cities */
   for (i=0; i<CityCnt; i++) {
      lat = cities[i].n_lat;
      lon = cities[i].w_lon;
      /* ignore cities whose longitude is not between From and To */
      if ( (!GT180 && ((lon > lon2) || (lon < lon1))) ||
           ( GT180 &&  (lon < lon2) && (lon > lon1)) ) continue;
      /* From and To are in the same hemisphere, ignore cities in the other */
      if ( (lat1 > 0) && (lat2 > 0) && (lat < lat1) ) continue;
      if ( (lat1 < 0) && (lat2 < 0) && (lat > lat2) ) continue;
      /* From and To are in different hemispheres, ignore polar routes */
      if ( (lat1 < 0) && (lat2 > 0) && ((lat > lat2) || (lat < lat1)) ) continue;
      if ((FIndex == i) || (TIndex == i)) continue;
      d1 = Dist(FIndex, i, MiFlag);
      d2 = Dist(TIndex, i, MiFlag);
      if (fabs(d1 + d2 - distance) < Delta) {
         WayPts[WayCnt].City = i;
         WayPts[WayCnt].Dist = d1;
         WayPts[WayCnt].Delta = (d1 + d2 - distance);
         if (++WayCnt == MAX_WAY_PT) break;
      }
   }
   SortWayPts();
}

xy_to_latlon(int x, int y, float *lat, float *lon) {

    /* alal */
    *lon = ((r_marg + l_marg) / 2 - x) * xspan / (r_marg - l_marg) + mid_lon;
    *lat = ((b_marg + t_marg) / 2 - y) * yspan / (b_marg - t_marg) + mid_lat;
}

LatLonToXY(int *x, int *y, float lat, float lon)
{
   *x = (l_marg + r_marg)/2 - ((lon - mid_lon) * (r_marg - l_marg) / xspan);
   *y = (t_marg + b_marg)/2 - ((lat - mid_lat) * (b_marg - t_marg) / yspan);
}

CityToXY(int *x, int *y, int Index)
{
   float lat, lon;

   lat = cities[Index].n_lat;
   lon = cities[Index].w_lon;
   LatLonToXY(x, y, lat, lon);

}

CrossHair(int x, int y, int len, int cross)
{
   SetRule(XOR_RULE);
   if (cross == 0) {
      Line(x-len, y, x+len, y);
      Line(x, y-len, x, y+len);
   }
   else {
      Line(x-len, y-len, x+len, y+len);
      Line(x-len+1, y-len, x+len+1, y+len);
      Line(x-len, y+len, x+len, y-len);
      Line(x-len+1, y+len, x+len+1, y-len);
   }
}

void BlinkEndPts(int FIndex, int TIndex, int WIndex)
{
   int x, y;
   static int FI = -1;
   static int TI = -1;
   static int WI = -1;

   blink_on = !blink_on;

   if (blink_on) {
     FI = FIndex;
     TI = (FIndex == TIndex) ? -1 : TIndex;
     WI = WIndex;
   }
   if (FI > -1 ) {
     CityToXY(&x, &y, FI);
     CrossHair(x, y, 4, 1);
   }
   if (TI > -1 ) {
     CityToXY(&x, &y, TI);
     CrossHair(x, y, 4, 1);
   }
   if (WI > -1) {
     CityToXY(&x, &y, WayPts[WI].City);
     CrossHair(x, y, 3, 0);
   }
}

ShowWayPts()
{
   int i, x, y;
   for (i=0; i<WayCnt; i++) {
     CityToXY(&x, &y, WayPts[i].City);
     CrossHair(x, y, 3, 0);
   }
}

void DisplayCityName(char *city, char *country)
{
  char str[STR_LEN];

  SetRule(FORCE_RULE);
  sprintf(str, "%s, %s", city, country);
#if 0
  /* alal */
  Text(5, 179, 480, str);
#else
  Text(5, 179, 513, str);
#endif
}

void UpdateMapDisplay(int CurCity, int FIndex, int TIndex, int *ShowWP, int *HideCity)
{
  char str[STR_LEN];

  if (*HideCity == 0)
    DisplayCityName(cities[CurCity].name, cities[cities[CurCity].country].cname);
  *HideCity = 0;

  if (*ShowWP) {ShowWayPts(); *ShowWP = 0;}
  if ((FIndex != oldFI) || (TIndex != oldTI)) WayCnt = 0;

  oldFI = FIndex;
  oldTI = TIndex;
}

void UpdateListDisplay(int CurCity, int FIndex, int TIndex, int WIndex,
   int MiFlag, int *UpdateAll, float *Distance, float *TripTotal,
   float *Delta)
{
   char str[STR_LEN], str1[STR_LEN];
   float Convert;
   int i;

   SetRule(FORCE_RULE);

   if (*UpdateAll) {
      Line(0, 60, 640, 60);
      Line(382, 60, 382, 190);
      Line(382, 110, 640, 110);
   }

   if ((FIndex != oldFI) || *UpdateAll) {
      if (FIndex < 0) str1[0] = '\0';
      else sprintf(str1, "%s, %s",
         cities[FIndex].name,
         cities[cities[FIndex].country].cname);
      sprintf(str, "From: %s", str1);
      Text(5, 15, 490, str);
      if (FIndex > -1)  sprintf(str, "(%2.1f,%3.1f)",
         cities[FIndex].n_lat, cities[FIndex].w_lon);
      else *str = '\0';
      Text(500, 15, 138, str);
   }

   if ((TIndex != oldTI) || *UpdateAll) {
      if (TIndex < 0) str1[0] = '\0';
      else sprintf(str1, "%s, %s",
         cities[TIndex].name,
         cities[cities[TIndex].country].cname);
      sprintf(str, "To:   %s", str1);
      Text(5, 28, 490, str);
      if (TIndex > -1) sprintf(str, "(%2.1f,%3.1f)",
         cities[TIndex].n_lat, cities[TIndex].w_lon);
      else *str = '\0';
      Text(500, 28, 138, str);
   }

   Convert = MiFlag ? MI_PER_KM : KM_PER_MI;

   if ((FIndex != oldFI) || (TIndex != oldTI)) WayCnt = 0;

   if (oldMiFlag != MiFlag)
     for (i=0; i<WayCnt; i++) {
        WayPts[i].Dist = WayPts[i].Dist * Convert;
        WayPts[i].Delta = WayPts[i].Delta * Convert;
     }

   if ((WIndex != oldWI) || (oldMiFlag != MiFlag) || *UpdateAll) {
      if (WayCnt > 0) {
         sprintf(str, "Via:  %s, %s",
            cities[WayPts[WIndex].City].name,
            cities[cities[WayPts[WIndex].City].country].cname);
      } else strcpy(str, "Via:");
      Text(5, 40, 490, str);
      if (WayCnt > 0) sprintf(str, "(%3.1f,%3.1f)",
         cities[WayPts[WIndex].City].n_lat, cities[WayPts[WIndex].City].w_lon);
      else *str = '\0';
      Text(500, 40, 138, str);
   }

   if (*UpdateAll) {
      LbClear(&TravelDlg, WAYPTS);
      IhListBox(IM_SHOW, 0, 0, &TravelDlg, &TravelDlgItems[WAYPTS]);
      LbSetCount(&TravelDlg, WAYPTS, (WayCnt == 0) ? 1 : WayCnt);
   }

   if ((FIndex != oldFI) || (TIndex != oldTI) || (oldMiFlag != MiFlag) || *UpdateAll) {
      if ((FIndex >= 0) && (TIndex >= 0))
         *Distance = Dist(FIndex, TIndex, MiFlag);
      else *Distance = 0.0;

      sprintf(str, "    Distance: %.1f %s", *Distance, MiFlag ? "mi" : "km");
      Text(392, 74, 240, str);
   }

   if (oldMiFlag != MiFlag)
      *TripTotal = *TripTotal * Convert;
   sprintf(str, "  Trip Total: %.1f", *TripTotal);
   Text(392, 88, 240, str);

   if ((oldMiFlag != MiFlag) || (*UpdateAll)) {
      GetDlgItem(&TravelDlg, DELTA, EDGI_TXT, str);
      sscanf(str, "%f", Delta);
      if (oldMiFlag != MiFlag)
        *Delta = *Delta * Convert;
      sprintf(str, "%.0f", *Delta);
      SetDlgItem(&TravelDlg, DELTA, EDSI_TXT, str);
   }

   sprintf(str, "     Way Pts: %d", WayCnt);
   Text(392, 138, 240, str);

   if (WayCnt > 0) sprintf(str, " Way Pt Dist: %.0f %.0f\%",
            WayPts[WIndex].Dist,
            (*Distance > 0) ? (WayPts[WIndex].Dist *100 / *Distance) : 0);
   else sprintf(str, " Way Pt Dist: ");
   Text(392, 152, 240, str);

   if (WayCnt > 0) sprintf(str, "Way Pt Delta: +%.0f", WayPts[WIndex].Delta);
   else sprintf(str, "Way Pt Delta: ");
   Text(392, 166, 240, str);

   if (CurCity > -1) LbSetPos(&TravelDlg, CITYS, CurCity);
   if (WayCnt > 0)   LbSetPos(&TravelDlg, WAYPTS, WIndex);

   oldFI = FIndex;
   oldTI = TIndex;
   oldWI = WIndex;
   oldMiFlag = MiFlag;
   *UpdateAll = 0;
}

int GetNum(char *Ptr)
{
   if ((*Ptr >= '0') && (*Ptr <= '9')) return (*Ptr - '0');
   else return -1;
}

void ShowMapFKeys(int SearchCities)
{
   /*strcpy(MapFnKeys[8], SearchCities ? "SCtry" : "SCity"); alal*/
   MapFnKeys[8] = SearchCities ? "SCtry" : "SCity";
   ShowFKeys(MapFnKeys);
   SelectFont(MEDIUM_FONT);
}

void ShowListFKeys(int SearchCities, int MiFlag)
{
   /*strcpy(ListFnKeys[2], MiFlag ? "Km" : "Mi");
   strcpy(ListFnKeys[8], SearchCities ? "SCtry" : "SCity"); alal*/
   ListFnKeys[2] = MiFlag ? "Km" : "Mi";
   ListFnKeys[8] = SearchCities ? "SCtry" : "SCity";
   ShowFKeys(ListFnKeys);
   SelectFont(MEDIUM_FONT);
}

void LoadMap(char *fname, int SearchCities)
{
   IMGHDR *pImg;

   pImg = LoadPcx(fname, 1);
   if(!pImg) FatalExit("Load PCX file failed", 1);

   PutImg(0, 0, FORCE_RULE, pImg);
   free(pImg);
   ShowTopTime(BANNER, 0);
   ShowMapFKeys(SearchCities);
}

PlotLine(double lat1, double lon1, double lat2, double lon2)
{
  double width, dtmp;
  int x1, y1, x2, y2;
  int lon1_vis, lon2_vis;

  SetRule(XOR_RULE);
  if (lon1 > mid_lon + 180) lon1 -= 360;
  if (lon1 < mid_lon - 180) lon1 += 360;
  if (lon2 > mid_lon + 180) lon2 -= 360;
  if (lon2 < mid_lon - 180) lon2 += 360;

  if (lon1 > lon2) {
     dtmp = lon1; lon1 = lon2; lon2 = dtmp;
     dtmp = lat1; lat1 = lat2; lat2 = dtmp;
  }

  if ((lon2 - lon1) > 180.) {
    /* wrap around */
    return (0);
  }

  lon1_vis = (lon1 > (mid_lon - xspan/2)) && (lon1 < (mid_lon + xspan/2));
  lon2_vis = (lon2 > (mid_lon - xspan/2)) && (lon2 < (mid_lon + xspan/2));

  LatLonToXY(&x1, &y1, lat1, lon1);
  LatLonToXY(&x2, &y2, lat2, lon2);

  width = 360 * (r_marg - l_marg) / xspan;

  if ((lon2 - lon1) < 180) Line(x1, y1, x2, y2);
  else if (!lon1_vis &&  lon2_vis) Line (x1+width, y1, x2, y2);
  else if ( lon1_vis && !lon2_vis) Line (x1, y1, x2-width, y2);
  else if (lon1_vis && lon2_vis) {
    Line(x1, y1, x2-width, y2);
    Line(x1+width, y1, x2, y2);
  }
}

PlotRoute(int FIndex, int TIndex)
{
  double lat1 = cities[FIndex].n_lat * DEG2RAD;
  double lon1 = cities[FIndex].w_lon * DEG2RAD;
  double lat2 = cities[TIndex].n_lat * DEG2RAD;
  double lon2 = cities[TIndex].w_lon * DEG2RAD;
  double latx, lonx, laty, lony, delta_lon, dtmp;
  int i;
  int pts = 16;
FILE *st;
st = fopen ("l", "a+t");

  if (fabs(lon1 - lon2) < .001) {
    PlotLine(lat1*RAD2DEG, lon1*RAD2DEG, lat2*RAD2DEG, lon2*RAD2DEG);
    return;
  }

  if (lon1 > lon2) {
     dtmp = lon1; lon1 = lon2; lon2 = dtmp;
     dtmp = lat1; lat1 = lat2; lat2 = dtmp;
  }

  if ((lon2 - lon1) > PI) delta_lon = ((lon2 - lon1) - (2 * PI)) / pts;
  else                    delta_lon =  (lon2 - lon1) / pts;
  if (delta_lon < -PI) delta_lon += 2*PI;
  if (delta_lon >  PI) delta_lon -= 2*PI;

  while ((fabs(delta_lon*RAD2DEG) < (2 * xspan/(r_marg - l_marg))) && (pts > 2)) {
    delta_lon *= 2;
    pts = pts/2;
  };

fprintf (st, "\nlon1 %7.2lf lon2 %7.2lf delta_lon %7.2lf\n", lon1, lon2, delta_lon);
fprintf (st, "mid_lon %7.2lf xspan %7.2lf\n", mid_lon, xspan);
  laty = lat1;
  lony = lon1;
  lonx = lon1 + delta_lon;
  SetClip(l_marg, t_marg, r_marg, b_marg);
  for (i=0; i<pts; i++) {
    latx = atan(((tan(lat2) * sin(lonx-lon1)) - (tan(lat1) * sin(lonx-lon2))) /
           sin(lon2-lon1));
fprintf (st, "lonx %7.2lf lony %7.2lf\n", lonx, lony);
    PlotLine(latx*RAD2DEG, lonx*RAD2DEG, laty*RAD2DEG, lony*RAD2DEG);
    laty = latx;
    lony = lonx;
    lonx = lonx + delta_lon;
  }
fclose (st);

  SetClip(0, 0, 639, 199);
}

WORD GetKbd(int FIndex, int TIndex, int WIndex)
{
   int count = 0;

   while (!KeyWaiting()) {
      MsDelay(120);
      count +=1;
      if(count%4 == 0) {
         BlinkEndPts(FIndex, TIndex, WIndex);
      }
      if (count == 100) {
         ShowTopTime(BANNER, 0);
         SelectFont(MEDIUM_FONT);
         SetColor(BLACK_COLOR);
         count = 0;
      }
   }
   return(GetKey());
}

void PrependCWD(char *fn)
{
  char cwd[STR_LEN];

  getcwd(cwd, STR_LEN);
  if (cwd[strlen(cwd)-1] != '\\') strcat(cwd, "\\");
  strcat(cwd, fn);
  strcpy(fn, cwd);
}

void NewMap(char *fname, int SearchCities)
{
  char *ptr;
  char pcxname[STR_LEN];
  char mapname[STR_LEN];
  int x1, y1, x2, y2;
  float lat1, lon1, lat2, lon2;

  if ((ptr = strstr(fname, ".")) != NULL) *ptr = '\0';
  strcpy(pcxname, fname);
  strcat(pcxname, ".pcx");
  PrependCWD(pcxname);
  strcpy(mapname, fname);
  strcat(mapname, ".map");
  PrependCWD(mapname);

  if (access(pcxname, 0) !=0 ) {putchar(BELL); return;}
  if (access(mapname, 0) !=0 ) {putchar(BELL); return;}
  strcpy(cur_map, fname);

  LoadMap(pcxname, SearchCities);
  fp = fopen(mapname,"rb");
  fscanf(fp, "%d %d %d %d %d %d %f %f %d %d %f %f",
    &t_marg, &b_marg, &l_marg, &r_marg, &x1, &y1, &lat1, &lon1, &x2, &y2, &lat2, &lon2);

  yspan = (b_marg - t_marg) * (lat1 - lat2) / (y2 - y1);
  if (x1 < x2)
   xspan = (r_marg - l_marg) * (((lon1 < lon2) ? lon1 + 360 : lon1) - lon2) / (x2 - x1);
  else
   xspan = (r_marg - l_marg) * (((lon2 < lon1) ? lon2 + 360 : lon2) - lon1) / (x1 - x2);
  mid_lat = lat1 + ((y1 - t_marg) * yspan / (b_marg - t_marg)) - (yspan/2);
  mid_lon = lon1 + ((x1 - l_marg) * xspan / (r_marg - l_marg)) - (xspan/2);
  if (mid_lon > 180) mid_lon -= 360;

  fclose(fp);
  blink_on = 0;
}

void MapDlg(int Registered, int SearchCities)
{
  char fname[STR_LEN], *ptr;
  struct ffblk ffblk;
  int Done = 1;
  int FirstTime = 1;
  int Key = 0;

  InitDialog(&MapNameDlg);
  ShowDialog(&MapNameDlg, 200, 75, Registered ? "Map Name" : "NOT REGISTERED!!!");
  strcpy(fname, "*.map");
  PrependCWD(fname);

  while(((Key &= 0x7f) != ESC_KEY) && (Key != ENTER_KEY)) {

    if (!Done) Done = findnext(&ffblk);
    if (Done) Done = findfirst(fname, &ffblk, 0);
    if (((Key == TAB_KEY) && !Done) || FirstTime) {
    if ((ptr = strstr(ffblk.ff_name, ".")) != NULL) *ptr = '\0';
      SetDlgItem(&MapNameDlg, MAP_NAME, EDSI_TXT, ffblk.ff_name);
      FirstTime = 0;
    }
    HandleDialog(&MapNameDlg, &Key);
  }

  GetDlgItem(&MapNameDlg, MAP_NAME, EDGI_TXT, &fname);
  CloseDialog(&MapNameDlg);
  if ((Key == ENTER_KEY) && Registered) NewMap(fname, SearchCities);
}

void EatWS(void)
{
  char c;

  while (((c=getc(fp)) == ' ') || (c=='\t') || (c=='\n') || (c==LF));
  ungetc(c, fp);
}

int GetStr(char *str) {
  int i;
  char c;

  EatWS();
  for(i=0; i<STR_LEN; i++) {
     if (((c=getc(fp)) == '/') || (c==LF) || (c=='\n') || (c==EOF)) break;
     else str[i] = c;
   }
  str[i] = '\0';
  return(i);
}

static void sysabort(
char *fmt, ...)
{
    va_list arg_ptr;

    fprintf(stderr,"\n\nabort: ");  /* preamble */
    va_start(arg_ptr, fmt);         /* arg_ptr -> first arg */
    vfprintf(stderr,fmt,arg_ptr);   /* display the message */
    va_end(arg_ptr);                /* arg_ptr = NULL */
    fprintf(stderr,"\n\n");         /* postamble */
    exit(0x7fff);                   /* do the abort */
}

static void *allocate(short nbytes)
{
    void *p;

    p = malloc(nbytes);
    if (p == NULL) sysabort("no more memory ...");
    return (p);
}

void GetCustomCities(int *CityCnt)
{
   char str[STR_LEN], fname[STR_LEN];
   float lat, lon;
   int i;
   CITY *c;

   strcpy(fname, "wtravel.dat");
   PrependCWD(fname);
   fp = fopen(fname,"rb");

   if (fp == NULL) return;
   for (i=*CityCnt; i<NC; i++) {
      c=&cities[i];
      if (GetStr(str) == 0) break;
      c->name = allocate(strlen(str)+1);
      strcpy(c->name, str);

      c->country = i;
      if (GetStr(str) == 0) break;
      c->cname = allocate(strlen(str)+1);
      strcpy(c->cname, str);

      if (GetStr(str) == 0) break;
      sscanf(str, "%f", &(c->n_lat));

      if (GetStr(str) == 0) break;
      sscanf(str, "%f", &(c->w_lon));

      /* added by alal */
      if (GetStr(str) == 0) break;
      sscanf(str, "%f", &(c->offset_hours));

      /* added by alal */
      if (GetStr(str) == 0) break;
      sscanf(str, "%c", &(c->dst_zone));

      (*CityCnt)++;
   }
   fclose(fp);
}

BombSight(WORD Key, int step) {
  static WORD LastKey;
  float lon, lat;
#if 0
alal
  static clock_t LastTime;
  clock_t ThisTime;
  char str[STR_LEN];
  int step;

  ThisTime = clock();
#else
  static time_t LastTime;
  time_t ThisTime;
  char str[STR_LEN];
/*  int step;*/

  ThisTime = time(NULL);
#endif
/*printf("%d, %d\n", ThisTime, LastTime);*/
#if 0
  if ((Key == LastKey) && ((ThisTime - LastTime) < 3)) {
    if (++samecnt > 3) {
      /* 10 pixel step only if same key withen 3 secs and after 5 times */
      step = 10;
    } else {
      step = 1;
    }
  } else {
    samecnt = 0;
    step = 1;
  }
#else
  if (step == 0) {
    step = ((Key == LastKey) && ((ThisTime - LastTime) < 3)) ? 10 : 1;
  }
#endif

  if (!FirstCH) CrossHair(CHx, CHy, 15, 0);

  if ((Key==HOME_KEY) && (CHx >= step)) CHx -= step;
  if ((Key== END_KEY) && (CHx <= (639-step))) CHx += step;
  if ((Key==PGUP_KEY) && (CHy >= (13+step))) CHy -= step;
  if ((Key==PGDN_KEY) && (CHy <= (177-step))) CHy +=step;

  LastKey = Key;
  LastTime = ThisTime;

  CrossHair(CHx, CHy, 15, 0);

  if (bombmode) {
    sprintf(str, "x:%3d, y:%3d", CHx, CHy);
  } else {
    /* alal */
    xy_to_latlon (CHx, CHy, &lat, &lon);
    sprintf(str, "%6.1f,%5.1f", lon, lat);
  }
  Text(518, 179, 130, str);
  FirstCH = 0;
}

void NearBy(int *CurCity, int CityCnt)
{
  int i, x, y;
  int MinCity = -1;
  unsigned long SqDistance;
  unsigned long MinDistance = -1;

  for(i=0; i<CityCnt; i++) {
    CityToXY(&x, &y, i);
    if ((x < l_marg) || (x > r_marg) || (y < t_marg) || (y > b_marg)) continue;
    SqDistance = ((long)(x-CHx)*(long)(x-CHx)) + ((long)(y-CHy)*(long)(y-CHy));
    if (SqDistance < MinDistance) {
      MinDistance = SqDistance;
      MinCity = i;
    }
  }
  if (MinCity >= 0) {
    *CurCity = MinCity;
    CityToXY(&x, &y, MinCity);
    CrossHair(x, y, 3, 0);
  }
}


double bearing2dest (double long4, double lat4, double long2, double lat2) {
/****************************************************************************
 *  compute bearing of 2 city from 4 city
 ****************************************************************************/
    double deg2rad, projy, projz, t;
    struct vec_p p2, p4, p4y, proj;
    struct vec_r r2, r4, r4y, r4z, projr;

    deg2rad = 3.141592653589793 / 180.;

    mk_po (p2, long2 * deg2rad, lat2 * deg2rad, 1.);
    mk_po (p4, long4 * deg2rad, lat4 * deg2rad, 1.);
    mk_po (p4y, (long4 + 90.) * deg2rad, 0., 1.);

    polar2rect (&p2, &r2);
    polar2rect (&p4, &r4);
    polar2rect (&p4y, &r4y);

    cross_p (&r4, &r4y, &r4z);
/*
    prn_vec_polar (&p4,  "p4",  NULL);
    prn_vec_polar (&p4y, "p4y", NULL);
    prn_vec_polar (&p2,  "p2",  NULL);
*/
/*
    prn_vec_rect  (&r4,  "r4",  NULL);
    prn_vec_rect  (&r4y, "r4y", NULL);
    prn_vec_rect  (&r4z, "r4z", NULL);
    prn_vec_rect  (&r2,  "r2",  NULL);
*/

    projy = dot_p (&r2,  &r4y);
    projz = dot_p (&r2,  &r4z);

    mk_rt (projr, projy, projz, 0.);

    rect2polar (&projr, &proj);
/*
    prn_vec_rect  (&projr, "projr", NULL);
    prn_vec_polar (&proj,  "proj",  NULL);
*/
/*  printf ("r4  . r4y %16.12lf\n", dot_p (&r4,  &r4y));
    printf ("r4  . r4z %16.12lf\n", dot_p (&r4,  &r4z));
    printf ("r4y . r4z %16.12lf\n", dot_p (&r4y, &r4z));
    printf ("r4  . r2  %16.12lf\n", dot_p (&r4,  &r2));
    printf ("r2  . r4y %16.12lf\n", dot_p (&r2,  &r4y));
    printf ("r2  . r4z %16.12lf\n", dot_p (&r2,  &r4z));*/

    t = proj.alpha / deg2rad - 90.;
    t = -t;
    if (t < 0.) {
        t += 360.;
    }
    if (t > 359.9999) {
        t = 0.;
    }

    return (t);
}


void main()
{
   char SearchStr[STR_LEN];
   char Str[STR_LEN];
   int CityCnt = 0;
   int Key = -1;
   int Focus = -1;
   int FIndex = -1;
   int TIndex = -1;
   int WIndex = 0;
   int CurCity = 0;
   int MiFlag = 1;
   int n;
   int MapMode = 1;
   int MapExit = 0;
   int DlgExit = 0;
   int UpdateAll = 1;
   int ShowWP = 0;
   int HideCity = 0;
   int Registered = 0;
   int SearchCities = 1;
   int bearing = 0;
   float Delta = 50;
   float TripTotal = 0;
   float Distance = 0;

   if(!PalInit(1)) FatalExit("Init failed - CGAGRAPH not loaded ?", 1);
   printf("One moment please...\n");
   GetCustomCities(&CityCnt);
   printf("Custom cities: %d\n", CityCnt);
   n = CityCnt;
   CityCnt = GetCityInfo(n);
   printf("Standard cities: %d\n", CityCnt - n);

   fp = fopen("c:\\_dat\\wtravel.env","rt");
   if (fp != NULL) {
     fscanf(fp, "%d %f %s", &MiFlag, &Delta, cur_map);
     oldMiFlag = MiFlag;
     fclose(fp);
   } else strcpy(cur_map, "world");

   NewMap(cur_map, SearchCities);
   ClearStr(SearchStr);

   strcpy(Str, "wtravel.reg");
   PrependCWD(Str);
   if (access(Str, 0) == 0) Registered = 1;

   do {
      if (MapMode) {
         UpdateMapDisplay(CurCity, FIndex, TIndex, &ShowWP, &HideCity);
         Key = GetKbd(FIndex, TIndex, (WayCnt > 0) ? WIndex : -1);
         if (Key & 0xff) Key &= 0xff;
         CitySearch(&CurCity, SearchStr, Key, CityCnt, SearchCities);
         switch (Key) {
            case F1_KEY:
               FIndex = CurCity;
               ClearStr(SearchStr);
               break;
            case F2_KEY:
               TIndex = CurCity;
               ClearStr(SearchStr);
               if ((FIndex >= 0) && (TIndex >= 0))
                  TripTotal += Dist(FIndex, TIndex, MiFlag);
               break;
            case F3_KEY:
               if ((FIndex >= 0) && (TIndex >= 0))
                  PlotRoute(FIndex, TIndex);
               break;
            case F4_KEY:
               if ((Delta >= 0) && (WayCnt == 0)) {
                 FindWayPt(FIndex, TIndex, CityCnt, Delta, MiFlag);
                 WIndex = 0;
               }
               ShowWayPts();
               break;
            case F5_KEY:
               NewMap(cur_map, SearchCities);
               FirstCH = 1;
               break;
            case F6_KEY:
               MapDlg(Registered, SearchCities);
               FirstCH = 1;
               break;
            case F7_KEY:
               NearBy(&CurCity, CityCnt);
               break;
            case F8_KEY:
               MapMode = 0;
               UpdateAll = 1;
               InitDialog(&TravelDlg);
               InitItems(CityCnt, Delta);
               ShowDialog(&TravelDlg, 0, 0, "World Travel");
               ShowListFKeys(SearchCities, MiFlag);
               break;
            case F9_KEY:
               SearchCities = !SearchCities;
               ClearStr(SearchStr);
               ShowMapFKeys(SearchCities);
               break;
            case F10_KEY:
               MapExit = 1;
               break;
            case UP_KEY:
               if (CurCity > 0) CurCity--;
               break;
            case DOWN_KEY:
               if (CurCity < CityCnt-1) CurCity++;
               break;
            case LEFT_KEY:
               if (WayCnt > 0) {
                  if (WIndex > 0) WIndex--;
                  strcpy(Str, "Via: ");
                  strcat(Str, cities[WayPts[WIndex].City].name);
                  DisplayCityName(Str,
                    cities[cities[WayPts[WIndex].City].country].cname);
                  HideCity = 1;
               }
               break;
            case RIGHT_KEY:
               if (WayCnt > 0) {
                  if (WIndex < WayCnt-1) WIndex++;
                  strcpy(Str, "Via: ");
                  strcat(Str, cities[WayPts[WIndex].City].name);
                  DisplayCityName(Str,
                    cities[cities[WayPts[WIndex].City].country].cname);
                  HideCity = 1;
               }
               break;

            case HOME_KEY:
            case END_KEY:
            case PGUP_KEY:
            case PGDN_KEY:
               BombSight(Key, 0);
               break;

            case '5':
               bombmode = !bombmode; /* toggle bombsight read out - alal */
               BombSight(0, 1); /* force update - alal */
               break;

            case '4':
               BombSight(HOME_KEY, 1);
               break;

            case '6':
               BombSight(END_KEY, 1);
               break;

            case '8':
               BombSight(PGUP_KEY, 1);
               break;

            case '2':
               BombSight(PGDN_KEY, 1);
               break;


            case '7':
               BombSight(HOME_KEY, 1);
               BombSight(PGUP_KEY, 1);
               break;

            case '9':
               BombSight(END_KEY, 1);
               BombSight(PGUP_KEY, 1);
               break;

            case '3':
               BombSight(END_KEY, 1);
               BombSight(PGDN_KEY, 1);
               break;

            case '1':
               BombSight(PGDN_KEY, 1);
               BombSight(HOME_KEY, 1);
               break;

            case ';':
               BombSight(HOME_KEY, 10);
               break;

            case '\'':
               BombSight(END_KEY, 10);
               break;

            case ']':
               BombSight(PGUP_KEY, 10);
               break;

            case '>':
               BombSight(PGDN_KEY, 10);
               break;


            case '[':
               BombSight(HOME_KEY, 10);
               BombSight(PGUP_KEY, 10);
               break;

            case '{':
               BombSight(END_KEY, 10);
               BombSight(PGUP_KEY, 10);
               break;

            case '?':
               BombSight(END_KEY, 10);
               BombSight(PGDN_KEY, 10);
               break;

            case '<':
               BombSight(PGDN_KEY, 10);
               BombSight(HOME_KEY, 10);
               break;

            default:
               break;
         }
      }
      else {
         UpdateListDisplay(CurCity, FIndex, TIndex, WIndex,
            MiFlag, &UpdateAll, &Distance, &TripTotal, &Delta);
/*alalal*/
         if ((bearing != 0) && (FIndex >= 0) && (TIndex >= 0)) {

            sprintf (Str, "Heading: %5.1lf                           ",
                bearing2dest (-cities [FIndex].w_lon, cities [FIndex].n_lat,
                              -cities [TIndex].w_lon, cities [TIndex].n_lat));
            Text (5, 40, 490, Str);
            bearing = 0;
         }
         HandleDialog(&TravelDlg, &Key);
         CurCity = LbGetPos(&TravelDlg, CITYS);
         if (WayCnt > 0) WIndex = LbGetPos(&TravelDlg, WAYPTS);

         Focus = GetFocus(&TravelDlg);

         if (Key & 0xff) Key &= 0xff;
         if (Focus == CITYS) CitySearch(&CurCity, SearchStr, Key, CityCnt, SearchCities);
            else ClearStr(SearchStr);

         if ((Key==F1_KEY) || (Key==F2_KEY) || (Key==F4_KEY)) UpdateAll = 1;
         switch (Key) {
            case F1_KEY:
               FIndex = CurCity;
               ClearStr(SearchStr);
               WayCnt = 0; WIndex = -1;
               break;
            case F2_KEY:
               TIndex = CurCity;
               ClearStr(SearchStr);
               if ((FIndex >= 0) && (TIndex >= 0))
                  TripTotal += Dist(FIndex, TIndex, MiFlag);
               WayCnt = 0; WIndex = -1;
/*alal*/
               bearing = 1;
               break;
            case F3_KEY:
               MiFlag = !MiFlag;
               ShowListFKeys(SearchCities, MiFlag);
               break;
            case F4_KEY:
               GetDlgItem(&TravelDlg, DELTA, EDGI_TXT, Str);
               sscanf(Str, "%f", &Delta);
               if (Delta >= 0) {
                  sprintf(Str, "Via:  One moment please...");
                  Text(5, 40, 490, Str);
                  FindWayPt(FIndex, TIndex, CityCnt, Delta, MiFlag);
                  ShowWP = 1;
                }
               if (WayCnt > 0) {
                  SetFocus(&TravelDlg, WAYPTS);
                  WIndex = 0;
               }
               break;
            case F5_KEY:
               TripTotal = 0;
               break;
            case F6_KEY:
               TripTotal += Distance;
               break;
            case F7_KEY: /*alal*/
#ifdef SUN_RISE_SET
                alsun (CurCity, WIndex, Focus, Str);
               break;
#endif
            case F8_KEY:
               GetDlgItem(&TravelDlg, DELTA, EDGI_TXT, Str);
               sscanf(Str, "%f", &Delta);
               CloseDialog(&TravelDlg);
               ShowMapFKeys(SearchCities);
               MapMode = 1;
               break;
            case F9_KEY:
               SearchCities = !SearchCities;
               ClearStr(SearchStr);
               ShowListFKeys(SearchCities, MiFlag);
               break;
            case F10_KEY:
               GetDlgItem(&TravelDlg, DELTA, EDGI_TXT, Str);
               sscanf(Str, "%f", &Delta);
               DlgExit = 1;
               break;
            default:
               break;
         }
      }
   } while (!MapExit && !DlgExit);

   fp = fopen("c:\\_dat\\wtravel.env","wt");
   if (fp != NULL) {
     fprintf(fp, "%d %.0f %s", MiFlag, Delta, cur_map);
     fclose(fp);
   }

   if (DlgExit) CloseDialog(&TravelDlg);
   PalDeInit(1);
}



void alsun (int CurCity, int WIndex, int Focus, char *Str) {
    char tbuf [STR_LEN];
    struct ffblk ffblk;
    int i, rs, civ, loop, now_mode;
    int Key = 0, ret, wc, sun1st, sunlst;
    time_t tp, now, last_key, trise, tset;
    struct tm t, *T;

    char dst [4];
    int day, city;
    int year,month;
    double daylen, civlen, j, alt, az, ra, dec, r, gmtoff;
    double rise, set, civ_start, civ_end, tsouth;
    extern void sun_RA_dec( double d, double *RA, double *dec, double *r );
    extern void radec2topo (
        double ra,
        double dec,
        double lon,
        double lat,
        double j2k_gmt,
        double *alt,
        double *az
    );

    if (CurCity < 0) return;

    InitDialog (&sundlg);
    ShowDialog (&sundlg, (640 - sundlg.Width) / 2, 15, "Sun Info");

    city = CurCity;
    if ((WIndex >= 0) && (Focus == WAYPTS)) {
       city = WayPts[WIndex].City;
    }

    SetDlgItem (&sundlg, CITYNAME, STSI_LBL,       cities[city].name);
    SetDlgItem (&sundlg, COUNTRY , STSI_LBL,       cities[cities[city].country].cname);
    SetDlgItem (&sundlg, LONG, STSI_LBL, al_f_hm (-cities[city].w_lon, '', '\''));
    SetDlgItem (&sundlg, LAT , STSI_LBL, al_f_hm ( cities[city].n_lat, '', '\''));




    time (&tp);
    last_key = tp;

    loop = 1;
    now_mode = 1;

    do {
        recalc :
        T = localtime (&tp);
        t = *T;

        gmtoff = cities[city].offset_hours;

        /*  localtime's isdst does not work */
        if ((t.tm_mon < 3) || (t.tm_mon > 9)) {
            /* before Apr or after oct */
            t.tm_isdst = 0;
        } else if (t.tm_mon == 3) {
            /* Apr */
            /* t.tm_wday == 0 is Sunday */
            /* DST starts 1st Sunday */

            /*  week constant wc = (35 + wday - mmday) % 7 */
            wc = (35 + t.tm_wday - t.tm_mday) % 7;
            sun1st = 7 - wc;

            if (t.tm_mday >= sun1st) {
                /*  DST */
                t.tm_isdst = 1;
            } else {
                /*  ST */
                t.tm_isdst = 0;
            }
        } else if (t.tm_mon == 9) {
            /* Oct */
            /* DST ends last Sunday */

            /*  week constant wc = (35 + wday - mmday) % 7 */
            wc = (35 + t.tm_wday - t.tm_mday) % 7;
            sun1st = 7 - wc;
            sunlst = sun1st + 28;
            if (sunlst > 31) sunlst -= 7;

            if (t.tm_mday < sunlst) {
                /*  DST */
                t.tm_isdst = 1;
            } else {
                /*  ST */
                t.tm_isdst = 0;
            }
        } else {
            /* DST */
            t.tm_isdst = 1;
        }


        i = 0;
        if (t.tm_isdst != 0) {
            if (cities[city].dst_zone == 'n') {
                gmtoff += 1;
                i = 1;
            }
        } else {
            if (cities[city].dst_zone == 's') {
                gmtoff += 1;
                i = 1;
            }
        }

        sprintf (tbuf, "Local Time = GMT %+3.1lf +%s",
                        gmtoff - (i ? 1 : 0), (i ? "1.0 (DST)" : "0.0 (ST)"));

        sprint_tm (Str, &t);
        SetDlgItem (&sundlg, DATETIME, STSI_LBL, Str);

        year = t.tm_year + 1900;
        month = t.tm_mon + 1;
        day = t.tm_mday;

        daylen  = day_length(year,month,day,-cities[city].w_lon,cities[city].n_lat);
        civlen  = day_civil_twilight_length(year,month,day,-cities[city].w_lon,cities[city].n_lat);

        rise       = 0;
        set        = 0;
        civ_start  = 0;
        civ_end    = 0;

        rs   = sun_rise_set   (year, month, day, -cities[city].w_lon, cities[city].n_lat,
                                      &rise, &set );
        civ  = civil_twilight (year, month, day, -cities[city].w_lon, cities[city].n_lat,
                                      &civ_start, &civ_end );

        rise       += gmtoff;
        set        += gmtoff;
        civ_start  += gmtoff;
        civ_end    += gmtoff;

        if (rise      >= 24.) rise      -= 24;
        if (set       >= 24.) set       -= 24;
        if (civ_start >= 24.) civ_start -= 24;
        if (civ_end   >= 24.) civ_end   -= 24;


        switch( rs ) {
            case 0:
                SetDlgItem (&sundlg, SUN_RISE, STSI_LBL, 2 + al_f_hm (rise,   ':', ' '));
                SetDlgItem (&sundlg, SUN_SET , STSI_LBL, 2 + al_f_hm (set,    ':', ' '));
                SetDlgItem (&sundlg, SUN_LEN , STSI_LBL, 2 + al_f_hm (daylen, ':', ' '));
                break;

            case +1:
                SetDlgItem (&sundlg, SUN_SET , STSI_LBL, "");
                SetDlgItem (&sundlg, SUN_LEN , STSI_LBL, "");
                SetDlgItem (&sundlg, SUN_RISE, STSI_LBL, "Sun above horizon");
                break;

            case -1:
                SetDlgItem (&sundlg, SUN_SET , STSI_LBL, "");
                SetDlgItem (&sundlg, SUN_LEN , STSI_LBL, "");
                SetDlgItem (&sundlg, SUN_RISE, STSI_LBL, "Sun below horizon");
                break;
        }

        switch( civ ) {
            case 0:
                SetDlgItem (&sundlg, CIVIL_RISE, STSI_LBL, 2 + al_f_hm (civ_start, ':', ' '));
                SetDlgItem (&sundlg, CIVIL_SET , STSI_LBL, 2 + al_f_hm (civ_end,   ':', ' '));
                SetDlgItem (&sundlg, CIVIL_LEN , STSI_LBL, 2 + al_f_hm (civlen,    ':', ' '));
                break;

            case +1:
                SetDlgItem (&sundlg, CIVIL_SET , STSI_LBL, "");
                SetDlgItem (&sundlg, CIVIL_LEN , STSI_LBL, "");
                SetDlgItem (&sundlg, CIVIL_RISE, STSI_LBL, "Sun above horizon");
                break;

            case -1:
                SetDlgItem (&sundlg, CIVIL_SET , STSI_LBL, "");
                SetDlgItem (&sundlg, CIVIL_LEN , STSI_LBL, "");
                SetDlgItem (&sundlg, CIVIL_RISE, STSI_LBL, "Sun below horizon");
                break;
        }

        j = tm_to_jd2k (&t);

        sun_RA_dec (j - gmtoff / 24., &ra, &dec, &r);
        radec2topo (ra, dec, -cities[city].w_lon, cities[city].n_lat, j - gmtoff / 24., &alt, &az);

        if (ra < 0) {
            ra += 360;
        }

        sprintf (Str,                "%s  ", al_f_hm ( az, '', '\''));
        sprintf (Str + strlen (Str), "%s",   al_f_hm (alt, '', '\''));
        SetDlgItem (&sundlg, POSITION, STSI_LBL, Str);


        while (kbhit () == 0) {
            /*  if no key press, check for recalc */
            time (&now);
            if (((double) now - last_key) > 2) {
                /*  print GMT calculation on realtime or keystroke */
                SetDlgItem (&sundlg, NOTE, STSI_LBL, tbuf);
                if (now_mode != 0) {
                    /*  now mode and last key was 2 sec ago */
                    time (&tp);
                    goto recalc;
                }
            }
        }
        /*  print GMT calculation on realtime or keystroke */
        SetDlgItem (&sundlg, NOTE, STSI_LBL, tbuf);

        ret = HandleDialog (&sundlg, NULL);


        /*  remember last recalc time */
        time (&last_key);

        if ((Key & 0xff) != 0) {
            Key &= 0xff;
        }

        /*  handle HandleDialog return code, i.e. button */
        switch (ret) {
            case BT_ADD_YEAR  :
                now_mode = 0;
                tp +=  31536000;
                continue;

            case BT_SUB_YEAR  :
                now_mode = 0;
                tp -=  31536000;
                continue;

            case BT_ADD_MONTH :
                now_mode = 0;
                tp +=  2592000;
                continue;

            case BT_SUB_MONTH :
                now_mode = 0;
                tp -=  2592000;
                continue;

            case BT_ADD_WEEK  :
                now_mode = 0;
                tp +=  604800;
                continue;

            case BT_SUB_WEEK  :
                now_mode = 0;
                tp -=  604800;
                continue;

            case BT_ADD_DAY   :
                now_mode = 0;
                tp += 86400;
                continue;

            case BT_SUB_DAY   :
                now_mode = 0;
                tp -= 86400;
                continue;

            case BT_ADD_HOUR  :
                now_mode = 0;
                tp +=  3600;
                continue;

            case BT_SUB_HOUR  :
                now_mode = 0;
                tp -=  3600;
                continue;

            case BT_ADD_MIN   :
                now_mode = 0;
                tp +=   300;
                continue;

            case BT_SUB_MIN   :
                now_mode = 0;
                tp -=   300;
                continue;

            case BT_NOW   :
                now_mode = 1;
                time (&tp);
                continue;

            case BT_RISET :
                now_mode = 0;
                now = tp - t.tm_hour * 3600. - t.tm_min * 60. - t.tm_sec;
                trise = now + ((time_t) (rise * 3600));
                tset  = now + ((time_t) (set  * 3600));
                tsouth = (trise + tset) / 2;
                if (tp < trise + 300.) {
                    tp = tsouth;
                } else if (tp > tset - 300.) {
                    tp = trise;
                } else {
                    tp = tset;
                }
                continue;


            case BT_CLOSE :
            case DN_OK :
            case DN_CANCEL :
                loop = 0;
                Key = 0;
                break;

            case DN_ACK :
                continue;
        }

    } while (loop != 0);



    CloseDialog(&sundlg);
} /* alsun */

