/* PDB2PHN converts the HP 200LX PDB database (phone book) in conjunction
 * with GDBIO so that the output can be transferred to a mobile phone
 * with WWW/LX, Robot/LX and the robot script "phone.scr".
 *
 * (c) 2001 by Daniel Hertrich http://www.daniel-hertrich.de/pdb2x
 * 
 * needed: GDBIO (http://www.palmtop.net/super.html)
 *         Post/LX from D&A Software (http://www.dasoft.com)
 *
 * usage:  prepare phone book file with
 *
 *           GDBIO C:\_DAT\PHONE.PDB > PHONE.CDF
 *
 *         Then do a
 *
 *           GDBIO C:\_DAT\PHONE.PDB /S | more
 *       
 *         This will show you the structure of your phone database.
 *         Determine the field numbers corresponding to the phone
 *         number field(s) you want to import into the phone.
 *         You can use up to 5 fields, only one field per entry
 *         is actually used. Also you need to know which filed number
 *         contains the name for each entry. It sill be 1 in most cases.
 *
 *         Use PHONE.CDF as input file for PDB2PHN and give the 
 *         field numbers you want to be scanned for name a numbers
 *         as arguments. The first field in the argument list 
 *         which contains a phone number will be used, together
 *         with the name of the entry, as output of PDB2PHN.
 *         If the scanned entry doesn't contain any number in one of
 *         the fields, the entry is skipped and PDB2PHN continues with 
 *         the next one
 *
 *         So your output will only contain entries to which a number
 *         exists in one or more of the given fields, and the order
 *         of the field numbers you give corresponds to their priorities.
 *
 *         So, call for example
 *
 *         PDB2PHN PHONE.CDF 1 5 3 4 > phone.phn
 *
 *         This will for every entry first scan field 5, then field 3,
 *         then field 4 for a valid number, and the first number it finds
 *         will be used for the output. Filed 1 is used for the name.
 *
 *         Say, one entry has the following fields:
 *
 *         Field  Name    Content
 *
 *         1      Name    Hertrich, Daniel
 *         2      email   daniel@somewhere.intheuniverse
 *         3      phone1  +49 30 12345678
 *         4      phone2  +49 177 7654321
 *         5      phone3  
 *         6      address 
 *
 *         You see, phone3, which is field 5, is empty, so the above 
 *         given call of PDB2PHN will create the following output:
 *
 *         Hertrich, Daniel|+491777654321
 *
 *         Since you redirect the output into another file, you will end
 *         up with a file containing entries which can directly be sent
 *         to the phone using the robot/lx script PHONE.SCR, which comes
 *         with the Robot/LX package.
 *
 */
 
#define VERSION "2.0"
#define YEAR "2002"
#define TOKLENGTH 80    // maximal length for a token, such as email, phone no, name of phone.cdf
#define LINELENGTH 1024 // maximal length of a line in cdf file
#define NAMELENGTH 17   // maximal allowed string length for the name in the
                        // phone book of the telephone (AT+CPBW=? (last number)) 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>


FILE *fpi;
int namecol, prio1, prio2, prio3, prio4, prio5;
int linecounter=0;
int i,j;
char testchar;
char line[LINELENGTH];
char name[TOKLENGTH];
char number[TOKLENGTH];
char shortnum[TOKLENGTH];
char *dup;

void hello(char *filename, int nc, int p1, int p2, int p3, int p4, int p5);
void usage (void);
char *extr_cdf_col(char *cdfline, int c);
void process_file(int namecol, int p1, int p2, int p3, int p4, int p5);

/* MAIN ============================================== */

int main(int argc, char *argv[])
{
  if (argc < 4) usage();

  namecol = atoi(argv[2]);

  prio1 = atoi(argv[3]);
  
  prio2 = prio3 = prio4 = prio5 = 0;
  
  if (argc > 4) prio2 = atoi(argv[4]); 
  if (argc > 5) prio3 = atoi(argv[5]);
  if (argc > 6) prio4 = atoi(argv[6]);
  if (argc > 7) prio5 = atoi(argv[7]);

  hello(argv[1], namecol, prio1, prio2, prio3, prio4, prio5);
  
  if ((fpi=fopen(argv[1],"r"))!=NULL)
  {

    process_file(namecol, prio1, prio2, prio3, prio4, prio5);
    
    if ((namecol == 0) && (prio1 == 0))
    {
      fprintf(stderr,"PDB2ADR:\a\nNothing to do, no valid column for name or phone number given!\n\n");
    }

    fclose(fpi);
  }
  else 
  {
    fprintf(stderr,"\n\aError opening input file \"%s\"!\nAbnormal program termination.\n\n",argv[1]);
    return(1);
  }
  return(0);
}



/* HELLO ============================================= */

void hello(char *filename, int nc, int p1, int p2, int p3, int p4, int p5)
{
  if (filename == NULL)
  {
    filename="NO INPUT FILE NAME GIVEN!";
  }
  fprintf(stderr,"\n*** This is PDB2PHN, Version %s, (c) 2002 Daniel Hertrich\n",VERSION);
  fprintf(stderr,"*** Input CDF file: %s\n",filename);
  fprintf(stderr,"*** Field numbers: Name: %i, Prio1: %i, Prio2: %i, Prio3: %i, Prio4: %i, Prio5: %i\n",nc,p1,p2,p3,p4,p5);
  fprintf(stderr,"*** Please wait while processing input file.....\n\n");
}

/* USAGE =========================================== */

void usage (void)
{
  printf("\n*** This is PDB2PHN, Version %s, (c) %s Daniel Hertrich\n",VERSION,YEAR);
  printf("\a\n\nError!\nCall of PDB2PHN: PDB2PHN <input CDF file> <name field> <f1 [f2 [f3 [f4 [f5] ] ] ] > > <output PHN file>\n");
  printf("See pdb2phn.c source file for details\n\n");
  exit(1);
}


/* EXTR_CDF_COL ===================================== */
/* give parameters:
      1. copy of a line of a cdf file (made by gdbio)
      2. number of column you want to extract
   get back:
      pointer to string containing string of given column
*/

char *extr_cdf_col(char *cdfline, int c)
{
  static char token[TOKLENGTH];
  char *loc;
  char *temptok = malloc(LINELENGTH*sizeof(char));
  int i = 0;

  temptok[0]=0;
    
  for (i=0; i<c; i++)
  {
    if (i>0) cdfline=loc+3;
    loc=strstr(cdfline,"\",\"");
    if (loc != NULL)
    { 
      strncpy(temptok,cdfline,loc-cdfline);
      temptok[loc-cdfline]='\0';
    }
    else
    {
      fprintf(stderr,"ERROR! Contact Author! loc is NULL!, c is %i\nCDFLINE is: %s\nLine number: %i\n\n\a\a",c,cdfline,linecounter);      
      exit(1);
    }
  }
  strcpy(token,temptok);
  free(temptok);
  return(token);
}


/* PROCESS_FILE ========================================= */

void process_file(int nc, int p1, int p2, int p3, int p4, int p5)
{
  while((testchar=fgetc(fpi))!= EOF)
  {
    linecounter++;
    fgets(line, LINELENGTH, fpi);
    dup=strdup(line);
    strncpy(name,extr_cdf_col(dup,nc),NAMELENGTH);
    strcpy(number,extr_cdf_col(line,p1));
    if (strlen(number) == 0)
    {
      strcpy(number,extr_cdf_col(line,p2));
      if (strlen(number) == 0)
      {
        strcpy(number,extr_cdf_col(line,p3));
        if (strlen(number) == 0)
        {
          strcpy(number,extr_cdf_col(line,p4));
          if (strlen(number) == 0)
          {
            strcpy(number,extr_cdf_col(line,p5));
          }
        }
      }
    }

    free(dup);
    if (strlen(number) > 0)  /* wenn in einem feld eine Nummber vorhanden war */
    {
      j = 0;
      strcpy(shortnum,"");
      for (i=0;i<strlen(number);i++)
      {
        if ((number[i]!=' ') && (number[i]!='\t') && (number[i]!='-') && (number[i]!='/') && (number[i]!='.'))
/* ersetzen durch Audruck "wenn nicht nummer oder +" */
        {
          shortnum[j] = number[i];
          j++;
          shortnum[j] = '\0';
        }
      }
      printf("%s|%s\n",shortnum,name);
    }
  }
}



