'/* TFIND.BAS Find matching text in a Binary-mode file */
'/*           By: Dale Thorn                           */
'/*           Rev. 28.04.2001                          */

' IN:   cbuf   If length >= tf.llof, cbuf must contain the entire file
'              Must be sent uppercased if NOT tf.csen and NOT tf.load
'       ctxt   Must be sent uppercased if NOT tf.csen and NOT tf.load
'    tf.buflen Main file buffer length (typically cbuf)
'    tf.txtlen Searchtext string length (typically ctxt)
'    tf.bbeg   Begin byte in file buffer for text search
'    tf.burp   EOF warning flag (valid only if wrap = TRUE)
'    tf.chr1   when nosp = TRUE, handle same as <= ASCII 32
'    tf.chr2   when nosp = TRUE, handle same as <= ASCII 32
'    tf.csen   Case-sensitivity flag (TRUE = case sensitive)
'    tf.fnum   File channel/unit number for binary-mode file
'    tf.lcnt   OK to count text lines if TRUE and NOT tf.wrap
'    tf.load   Load initial buffer at fbeg position when TRUE
'    tf.lsav   Saved line length when c/r not found in buffer
'    tf.nosp   Eliminate spaces in file buffer for search purposes
'    tf.wrap   Wraparound search flag (terminate at fbeg + bbeg - 1)
'    tf.wwrd   Whole word search flag (substring match is not valid)
'    tf.fbeg   Begin byte in file corresponding to begin byte of buffer
'    tf.llof   Length of the binary-mode file to be searched
'
' OUT:  Function return (ifn.tfnd) = User-pressed ESC, or zero if no ESC
'       cbuf   Current buffer if text was found (contents may be uppercased)
'              Same as buffer sent if text not found, except not uppercased
'    tf.bbeg   No find = 0, else byte# in cbuf where ctxt found
'    tf.llen   maximum length of any text line in the text file
'    tf.fbeg   Byte# in file corresponding to first byte in cbuf
'    tf.ltot   #text lines between bbeg and text find (if tf.lcnt)

dim tf as tfnd                         'pass-structure for text-search function
cwrdchr = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"

function ifn.tfnd(tf as tfnd, cbuf, ctxt)                 'text-search function
   tf.buflen = len(cbuf)                     'length of main file buffer (cbuf)
   tf.txtlen = len(ctxt)                    'length of searchtext string (ctxt)
   cchr1 = " "                           'initialize single-byte temp character
   ibrp = 0                                 'initialize the EOF User-alert flag
   ieof = 0                               'initialize search-went-past-EOF flag
   ikey = 0                                    'initialize the return-key value
   inxt = 0                                 'next search-begin position in cbuf
   ipos = 0                                 'textmatch position pointer in cbuf
   iptr = tf.bbeg - 1                    'initialize the line-count c/r pointer
   lptr = tf.fbeg                         'file position for begin byte of cbuf
   ltmp = 0                                   'buffer spans beyond EOF (if > 0)

   if tf.load then                          'OK to load the initial file buffer
      tf.load = 0                            'turn load-initial-buffer flag OFF
      i = ifn.tfnd.load(tf, cbuf, ltmp, tf.csen)'load the first-time file buffer
      if not tf.csen then                     'the case-sensitive switch is OFF
         lset ctxt = ucase$(ctxt)             'uppercase the text-search string
      end if
   end if
   do                                              'begin loop to find the text
      if tf.nosp then                              'eliminate-spaces flag is ON
         ipos = ifn.tfnd.nosp(tf, cbuf, ctxt)      'perform no-spaces text find
      elseif tf.wwrd then                         'whole-word search flag is ON
         do                                       'begin loop for normal search
            ipos = instr(tf.bbeg, cbuf, ctxt)     'perform a normal text search
            itmp = ipos                            'save the text search result
            if ipos then                           'part-word searchtext found!
               i = ifn.tfnd.wwrd(tf, cbuf, ctxt, ipos)   'match whole word only
            end if
            if itmp <> ipos then         'the whole-word match was disqualified
               tf.bbeg = itmp + 1      'reset search begin to the next position
            else                 'whole-word flag not on or no disqualification
               exit do           'flag not on or no disqualification; exit loop
            end if
         loop
      else                         'eliminate-spaces & whole-word flags are OFF
         ipos = instr(tf.bbeg, cbuf, ctxt)        'perform a normal text search
      end if
      if tf.lcnt and not tf.wrap then                   'OK to count text lines
         do                                     'begin loop to count text lines
            inxt = instr(iptr + 1, cbuf, char(13))  'posn.of next c/r character
            if inxt > 0 and (inxt < ipos or ipos = 0) then 'c/r character found
               itmp = inxt - iptr - 1 + tf.lsav      'total current line length
               if tf.llen < itmp then        'maximum linelen < current linelen
                  tf.llen = itmp             'set max.linelen = current linelen
               end if
               tf.ltot = tf.ltot + 1        'increment the number of text lines
               tf.lsav = 0                  'clear saved length of current line
               iptr = inxt                 'save current c/r character position
               if iptr < tf.buflen then     'line-count pointer < end of buffer
                  lset cchr1 = char(midchar(cbuf, iptr + 1))'poss.l/f character
               else                         'line-count pointer = end of buffer
                  get tf.fnum, tf.fbeg + iptr, cchr1    'possible l/f character
               end if
               if midchar(cchr1, 1) = 10 then    'c/r-l/f character pair found!
                  iptr = iptr + 1               'advance the line-count pointer
               end if
            else                               'c/r character NOT found in cbuf
               tf.lsav = tf.buflen - iptr     'saved length of the current line
               exit do                      'c/r character NOT found; exit loop
            end if
         loop
      end if
      if ipos then                   'matching text was found in current buffer
         if ibrp then                          'the EOF User-alert flag was set
            beep                        'signal that the search went beyond EOF
         end if
         exit do                             'matching text found; exit routine
      else                           'matching text NOT found in current buffer
         if tf.buflen >= tf.llof then       'file fits completely in one buffer
            if tf.bbeg = 1 or not tf.wrap then      'no wrap or already wrapped
               exit do         'file completely searched; no find; exit routine
            end if
            ibrp = tf.burp                         'set the EOF User-alert flag
         else                                  'file spans more than one buffer
            inxt = tf.buflen - tf.txtlen * 2   'next search-begin posn. in cbuf
            if inxt < tf.bbeg then          'search-begin position <= last find
               inxt = tf.bbeg            'set search-begin position > last find
            end if
            tf.fbeg = tf.fbeg + inxt - 1    'move file pointer to next position
            iptr = tf.buflen - inxt + 1 'set line-count pointer to buffer begin
            if tf.fbeg >= lptr and ieof then    'pointer is back to begin & EOF
               exit do         'file completely searched; no find; exit routine
            end if
            if tf.fbeg > tf.llof then           'the file pointer is beyond EOF
               if not tf.wrap then           'the wraparound-search flag is OFF
                  exit do                 'EOF; wraparound is OFF; exit routine
               end if
               ibrp = tf.burp                      'set the EOF User-alert flag
               ieof = not 0                'set the search-went-beyond-EOF flag
               tf.fbeg = 1               'set file pointer to beginning of file
            end if
            i = ifn.tfnd.load(tf, cbuf, ltmp, tf.csen)'get the next file buffer
            if ltmp > 0 and not tf.wrap then           'buffer spans beyond EOF
               ibrp = tf.burp                      'set the EOF User-alert flag
               ieof = not 0                'set the search-went-beyond-EOF flag
            end if
         end if
         tf.bbeg = 1           'set text-search position to beginning of buffer
      end if
      if inkey$ = char(27) then                   'the User pressed the ESC key
         ikey = 27                           'set return value to ESC key value
         exit do                               'search terminated; exit routine
      end if
   loop
   if ipos then                                'matching text was found in cbuf
      tf.bbeg = ipos                'return position for beginning byte of ctxt
   else                            'matching text NOT found or User pressed ESC
      tf.bbeg = 0                     'return zero; no find or User pressed ESC
      tf.fbeg = lptr           'restore pos. in file for beginning byte of cbuf
      i = ifn.tfnd.load(tf, cbuf, ltmp, -1)   'restore the original file buffer
   end if
   ifn.tfnd = ikey                       'return User-pressed ESC key (or zero)
end function

function ifn.tfnd.load(tf as tfnd, cbuf, ltmp, icas) 'load and prep main buffer
   get tf.fnum, tf.fbeg, cbuf                        'load the main file buffer
   if fre("") > tf.buflen then            'available free memory > buflen bytes
      imem = tf.buflen \ 3                'set the maximum blank segment length
   else                                  'available free memory <= buflen bytes
      imem = cint(fre("")) \ 3            'set the maximum blank segment length
      if imem < 1024 then                  'memory too low to process data load
         i = ifn.msgs("Not enough memory to load data", 5, 24, 79, 1, 1)
      end if                        'display the error message [above] and exit
   end if
   ltmp = tf.fbeg + tf.buflen - tf.llof - 1    'buffer spans beyond EOF (if >0)
   if ltmp > 0 then                                    'buffer spans beyond EOF
      for i = tf.llof - tf.fbeg + 2 to tf.buflen step imem   'clear unused data
         mid$(cbuf, i) = string$(imem, 0)         'clear buffer area beyond EOF
         if tf.buflen - i < imem then          'current segment is last segment
            exit for                         'last segment processed; exit loop
         end if
      next
   end if
   if not icas then                           'the case-sensitive switch is OFF
      for i = 1 to tf.buflen step imem          'loop thru the main file buffer
         mid$(cbuf, i) = ucase$(mid$(cbuf, i, imem))  'uppercase buffer segment
         if tf.buflen - i < imem then          'current segment is last segment
            exit for                         'last segment processed; exit loop
         end if
      next
   end if
end function

function ifn.tfnd.nosp(tf as tfnd, cbuf, ctxt)   'perform a no-spaces text find
   ipos1 = tf.bbeg - 1                'initialize the begin text match position
   do                              'loop thru the characters in the text buffer
      ipos1 = instr(ipos1 + 1, cbuf, char(midchar(ctxt, 1)))'get 1st text match
      if ipos1 = 0 or tf.txtlen = 1 then  'match not found or match 1 character
         exit do                      'end of search; return to calling routine
      end if
      ilen = 1                                  'match on 1 character so far...
      for ipos2 = ipos1 + 1 to tf.buflen'loop from first match to end of buffer
         ichr = midchar(cbuf, ipos2)      'character at current buffer position
         if ichr = midchar(ctxt, ilen + 1) then'match on next char.in searchtxt
            ilen = ilen + 1                'increment no. of characters matched
            if ilen = tf.txtlen then       'all characters in searchtxt matched
               exit do                'end of search; return to calling routine
            end if
         elseif not(ichr <= 32 or ichr = tf.chr1 or ichr = tf.chr2) then
            exit for         'no match on subsequent character; continue search
         end if
      next
   loop
   ifn.tfnd.nosp = ipos1      'return position of text match to calling routine
end function

function ifn.tfnd.wwrd(tf as tfnd, cbuf, ctxt, ipos)  'match on whole word only
   cchr1 = " "                       'initialize 1st single-byte temp character
   cchr2 = " "                       'initialize 2nd single-byte temp character
   if ipos > 1 then                   'previous byte within current text buffer
      lset cchr1 = char(midchar(cbuf, ipos - 1)) 'get previous byte from buffer
   elseif tf.fbeg > 1 then                'previous byte available in text file
      get tf.fnum, tf.fbeg - 1, cchr1     'get previous byte from the text file
   end if
   if ipos + tf.txtlen <= tf.buflen then  'next byte within current text buffer
      lset cchr2 = char(midchar(cbuf, ipos + tf.txtlen))'get next byte f/buffer
   elseif tf.fbeg + tf.buflen <= tf.llof then 'next byte available in text file
      get tf.fnum, tf.fbeg + tf.buflen, cchr2 'get next byte from the text file
   end if
   if instr(ctxt, cchr1) or instr(ctxt, cchr2) then 'valid word character found
      ipos = 0                  'previous byte==word char; word found NOT valid
   elseif instr(cwrdchr, cchr1) or instr(cwrdchr, cchr2) then 'word char. found
      ipos = 0                      'next byte==word char; word found NOT valid
   end if             'NOTE: use COMMON SHARED (cwrdchr) variable in above test
end function
