/***********************************************************************
RIE-EX2.C: Remote Import/Export example program #2
by MSW
Copyright (c) 1992 by cc:Mail, Inc.  All rights reserved.

This source code, any derivatives of it, and the resulting object code
may be freely used and/or copied, so long as this copyright notice is
included and remains unmodified.  cc:Mail, Inc. does not make any
representation or warranty, express or implied with respect to this
source code.  cc:Mail, Inc. disclaims any liability whatsoever for any
use of such code.

************************************************************************

    This program is an example of how to parse a Remote Import/Export
    DIRECTORY format file.  The key point to notice is how lines with
    field codes (e.g. "Addr:") that are not understood (e.g. "New1:")
    are ignored rather than considered an error.

    The program filters an input Remote I/E DIRECTORY format file,
    producing an output Remote I/E DIRECTORY format file that contains
    only those entries whose address matches the specified post office.
    The usage is:
        rie-ex2 addr [infile [outfile]]

            where addr    is the post office
                  infile  is the input file (or use standard input)
                  outfile is the output file (or use standard output)

    RIE-EX2 returns an ERRORLEVEL of 0 for success and 1 for any error.

    Build with Microsoft C version 5.1 or later using any memory model.
***********************************************************************/


// Include file(s)
#include <stdio.h>
#include <string.h>

// Define constants
#define BUFFSIZE    1024        // Size of buffer for each name

// Declare prototypes
int filt(char *buff, char *addr);
int match(char *buff, char *patt);


/***********************************************************************
RIE-EX2 main routine (described above)
***********************************************************************/
int main(int argc, char *argv[])
{
    char buff[BUFFSIZE], tmpChr;
    FILE *fpIn, *fpOut;
    int rtn;
    unsigned buffUsed;

    // Check argument count
    if (argc < 2  ||  4 < argc) {
        fprintf(stderr, "Usage: rie-ex2 addr [infile [outfile]]\n");
        return  1;
    }

    // Open output file (text mode)
    if (argc >= 4) {
        if ((fpOut = fopen(argv[3], "wt")) == NULL) {
            fprintf(stderr, "rie-ex2: Error opening %s\n", argv[3]);
            return  1;
        }
    }
    else {
        fpOut = stdout;
    }

    // Open input file (text mode)
    if (argc >= 3) {
        if ((fpIn = fopen(argv[2], "rt")) == NULL) {
            if (fpOut != stdout) {
                fclose(fpOut);
            }
            fprintf(stderr, "rie-ex2: Error opening %s\n", argv[2]);
            return  1;
        }
    }
    else {
        fpIn = stdin;
    }

    // Loop for input, one line at a time...
    buffUsed = 0;
    rtn = 0;
    for (;;) {
        if (fgets(buff + buffUsed, BUFFSIZE - buffUsed, fpIn) == NULL) {
            if (ferror(fpIn)) {
                fprintf(stderr, "rie-ex2: Error reading input file\n");
                rtn = 1;
                break;
            }

            // End of input: does final record meet filter requirements?
            if (filt(buff, argv[1])) {
                if (fputs(buff, fpOut) == EOF) {
                    fprintf(stderr,
                        "rie-ex2: Error writing output file\n");
                    rtn = 1;
                }
            }
            break;
        }

        // If we have started a new record, see if the previous record
        //    meets the filter requirements, and then move the new
        //    record to the start of the buffer
        if (match(buff + buffUsed, "name:")) {
            tmpChr = buff[buffUsed];
            buff[buffUsed] = '\0';
            if (filt(buff, argv[1])) {
                if (fputs(buff, fpOut) == EOF) {
                    fprintf(stderr,
                        "rie-ex2: Error writing output file\n");
                    rtn = 1;
                    break;
                }
            }
            buff[buffUsed] = tmpChr;
            memmove(buff, buff + buffUsed, BUFFSIZE - buffUsed);

            // Keep track of the how much of the buffer is used and loop
            //    for the next input line
            buffUsed = strlen(buff);
        }
        else {
            buffUsed += strlen(buff + buffUsed);
        }
    }

    // Close files (ignore errors) and exit
    if (fpIn != stdin) {
        fclose(fpIn);
    }
    if (fpOut != stdout) {
        fclose(fpOut);
    }
    return  rtn;
} // main


/***********************************************************************
Routine    : filt
Description: Determine if the specified buffer meets the specified
           : filter pattern.  In this example, the filter is met if the
           : address ("Addr:") starts with the pattern (the post office
           : name), and is followed by a newline, the end of the buffer,
           : or a space.  The post office name comparison is
           : case-insensitive.
Inputs     : buff       buffer to check for filter pattern in
           :            (null-terminated)
           : patt       pattern to use for filter (null-terminated)
Returns    : Zero (false) if the buffer does not meet the filter
***********************************************************************/
int filt(char *buff, char *patt)
{
    unsigned pattLen;

    // Loop for each line in the buffer (assume the address isn't the
    //    first line, and that the filter fails if no address is found)
    for (;;) {
        while (*buff != '\n') {
            if (*buff++ == '\0') {
                return  0;
            }
        }

        // If this is the address line, see if the filter pattern is met
        if (match(++buff, "addr:")) {
            buff += 5;          // Skip "addr:"
            while (*buff == ' ') {
                buff++;
            }
            pattLen = strlen(patt);
            if (!strnicmp(buff, patt, pattLen)  &&
                    (buff[pattLen] == '\n'  ||
                     buff[pattLen] == '\0'  ||
                     buff[pattLen] == ' '     )   ) {
                return  1;
            }
            return  0;
        }
    }
} // filt


/***********************************************************************
Routine    : match
Description: Determine if the specified buffer matches the specified
           : pattern.  There is a match if the pattern is at the start
           : of the buffer (after skipping leading spaces).  The
           : comparison is case-insensitive.  The characters in the
           : buffer after the pattern are ignored.
Inputs     : buff       buffer to check for pattern in (null-terminated,
           :            leading spaces are skipped)
           : patt       pattern to check for (null-terminated)
Returns    : Zero (false) if the match is not found
***********************************************************************/
int match(char *buff, char *patt)
{
    while (*buff == ' ') {
        buff++;
    }
    return  !strnicmp(buff, patt, strlen(patt));
} // match

// End of RIE-EX2.C

