#include "syncdir.h"
#include <assert.h>
#include <string.h>

#define TMPBUFSZ (60 * 1024)

/********************************************************************
 **
 ** FILETABLE
 **
 ********************************************************************/
typedef struct tagSUBFILETABLE
{
  UINT32    offset;    /* offset the start of this table     */
  UINT32    lastOfs;   /* offset the last entry read         */
  UINT32    nextOfs;   /* offset of the next entry to read   */
  UINT32    entryCt;   /* number of entries                  */
  UINT32    entryNo;   /* last accessed entry                */
  UINT32    entrySzMax;/* max # of bytes needed for an entry */
  FILEINFO *fi;        /* entry into temp buffer             */
} SUBFILETABLE;

typedef struct tagFT
{
  FILETABLE     ft;         /* must be first entry!                */
  FILE         *tmpFile;    /* temp caching file                   */
  char         *tmpBuf;     /* 60K buffer                          */
  FILEINFO    **idx;        /* current index entry                 */
  char         *fiPtr;      /* current file table entry            */
  SUBFILETABLE *subFT;      /* sub file table entry                */
  SUBFILETABLE *sub;        /* current entry into SFT              */
  UINT32        subFTalloc; /* # of sub file table entries alloc'd */
  UINT32        subFTct;    /* # of sub file table entries in use  */
} FT;

/*
small-memory file table
(only requires 64K + misc)

  1. alloc a 60K buffer
     index builds down, data builds up
     when sort, dump, add file location + entry count to sub-table index
*/
FILETABLE *fileTableAlloc(const char *name)
{
  FT *ft = MALLOC(FT);
  if (ft)
  {
    memset(ft, 0, sizeof(*ft));
    strcpy(ft->ft.name, name);
    ft->tmpFile = fileTempCreate();
    if (ft->tmpFile)
    {
      ft->tmpBuf = REALLOC(char, 0, TMPBUFSZ);
      if (ft->tmpBuf)
      {
        ft->fiPtr = ft->tmpBuf;
        ft->idx   = (void *) (ft->tmpBuf + TMPBUFSZ);
      }
    }
    if (!ft->tmpFile || !ft->tmpBuf)
      longjmp(jmpbuf, SYNCDIR_ERROR_TEMP);
  } else
    longjmp(jmpbuf, SYNCDIR_ERROR_MEMORY);
  return (FILETABLE *) ft;
}

void fileTableFree(FILETABLE *tbl)
{
  FT *ft = (FT *) tbl;
  free(ft->subFT);
  free(ft->tmpBuf);
  fileTempDestroy(ft->tmpFile);
  free(ft);
}

static void sort(FT *ft)
{
  UINT32 m;
  for (m = ft->sub->entryCt / 2; m; m /= 2)
  {
    UINT32 ii;
    for (ii = 0; ii + m < ft->sub->entryCt; ii++)
    {
      UINT32 jj;
      for (jj = ii; (fileInfoNameCompare(ft->idx[jj], ft->idx[jj+m], FALSE) > 0); jj -= m)
      {
        FILEINFO *tmp = ft->idx[jj];
        ft->idx[jj] = ft->idx[jj+m];
        ft->idx[jj+m] = tmp;
        if (jj < m) /* since jj cannot become negative, i need this test! */
          break;
      }
    }
  }
}

/*
  dump current sub table to file
*/
static void dump(FT *ft)
{
  UINT32 ii;
  if (ft->sub)
  {
    ft->sub->offset = ftell(ft->tmpFile);
    sort(ft);
    for (ii = 0; ii < ft->sub->entryCt; ii++)
    {
      FILEINFO *fi = ft->idx[ii];
      int       sz = sizeof(*fi) + fi->nameSz;
      fwrite(fi, sz, 1, ft->tmpFile);
    }
  }
}

BOOL fileTableAdd(FILETABLE *tbl, FILEINFO *fi)
{
  tbl->entryCt++;
  return TRUE;
}

static void getNextEntry(FT *ft, SUBFILETABLE *sub)
{
  char *buf   = (void *) sub->fi;
  if (!ft->sub)
  {
    sub->entryNo = 0;
    sub->lastOfs = sub->offset;
  } else if (++sub->entryNo == sub->entryCt)
    return;
  else
    sub->lastOfs = sub->nextOfs;

  fseek(ft->tmpFile, sub->lastOfs, SEEK_SET);
  fread(buf, sizeof(FILEINFO), 1, ft->tmpFile);                   /* get fixed bit */
  fread(buf + sizeof(FILEINFO), sub->fi->nameSz, 1, ft->tmpFile); /* get name      */
  sub->nextOfs = ftell(ft->tmpFile);
}

static FILEINFO *getSmallestEntry(FT *ft)
{
  UINT32        ii;
  if (!ft->sub)
  {
    /*
    read first entry of each table
    */
    for (ii = 0; ii < ft->subFTct; ii++)
      getNextEntry(ft, ft->subFT + ii);
  } else 
    getNextEntry(ft, ft->sub);
  for (ii = 0, ft->sub = 0; ii < ft->subFTct; ii++)
  {
    SUBFILETABLE *st = ft->subFT + ii;
    if (st->entryNo < st->entryCt)
    {
      if (ft->sub)
      {
        if (fileInfoNameCompare(ft->sub->fi, st->fi, FALSE) > 0)
          ft->sub = st;
      } else
        ft->sub = ft->subFT + ii;
    }
  }
  return (ft->sub) 
    ? ft->sub->fi
    : 0;
}

/*
when getting first:
  if this is the first call to get first
    dump the current subtable
    divy up the 60K buffer to give each subtable 1 entry
  set all subtable entryno to 0
  read the first entry from each sub table
  return the smallest entry
*/
FILEINFO *fileTableGetFirst(FILETABLE *tbl)
{
  FT *ft = (FT *) tbl;
  UINT32 ii;

  if (ft->idx)
  {
    char *ptr = ft->tmpBuf;
    dump(ft); /* dump the current subtable */
    ft->idx = 0;
    for (ii = 0; ii < ft->subFTct; ii++)
    {
      ft->subFT[ii].fi = (void *) ptr;
      ptr += ft->subFT[ii].entrySzMax;
    }
  }
  for (ii = 0; ii < ft->subFTct; ii++)
    ft->subFT[ii].entryNo = 0;
  ft->sub = 0;

  return getSmallestEntry(ft);
}

/*
when getting next
  read the next entry from the last subtable returned
  return the smallest subtable entry
*/
FILEINFO *fileTableGetNext(FILETABLE *tbl)
{
  FT *ft = (FT *) tbl;
  return (ft->sub)
    ? getSmallestEntry(ft)
    : 0;
}

/********************************************************************
 **
 ** FILEINFO
 **
 ********************************************************************/
FILEINFO *fileInfoAlloc(FILETABLE *tbl, const char *name)
{
  FT       *ft = (FT *) tbl;
  FILEINFO *fi;
  UINT32    fiSz = sizeof(FILEINFO) + strlen(name);

  if (!ft->subFT || ((char *) (ft->idx-1) < ft->fiPtr + fiSz))
  {
    /*
      either we've not yet allocated the buffer, or this entry doesn't fit
    */
    if (ft->subFT)
    {
      sort(ft);
      dump(ft);
    }
    if (ft->subFTct == ft->subFTalloc)
    {
      SUBFILETABLE *sFT = REALLOC(SUBFILETABLE, ft->subFT, ft->subFTalloc + 16);
      if (!sFT)
        longjmp(jmpbuf, SYNCDIR_ERROR_MEMORY);
      ft->subFT = sFT;
      ft->subFTalloc += 16;
    }
    /*
    reset pointers
    */
    ft->fiPtr= ft->tmpBuf;
    ft->idx  = (void *) (ft->tmpBuf + TMPBUFSZ);

    ft->sub = ft->subFT + ft->subFTct;
    ft->subFTct++;
    ft->sub->entryCt = 0;
    ft->sub->entrySzMax = fiSz;
  } else
    ft->sub->entrySzMax = max(ft->sub->entrySzMax, fiSz);
  fi = (FILEINFO *) ft->fiPtr;
  ft->fiPtr += fiSz;
  *(--ft->idx) = fi;
  ft->sub->entryCt++;
  fi->refCt = 1; /* do not want this free'd */
  fi->owner = tbl;
  fi->nameSz = strlen(name);
  fi->action = ACTION_NONE;
  fi->size   = 0;
  strcpy(fi->name, name);
  return fi;
}

void fileInfoFree(FILEINFO *inf)
{
  if (inf->refCt == 0)
    free(inf);
  else
    inf->refCt--;
}

BOOL fileInfoIsDirectory(const FILEINFO *inf)
{
  return inf->isDir;
}

BOOL fileInfoSetAction(FILEINFO *inf, ACTION act)
{
  FT *tbl = (FT *) inf->owner;
  inf->action = act;
  fseek(tbl->tmpFile, tbl->sub->lastOfs, SEEK_SET);
  fwrite(inf, sizeof(*inf), 1, tbl->tmpFile);
  fileInfoUpdateStats(inf);
  return TRUE;
}

FILEINFO *fileInfoDup(FILEINFO *r)
{
  FILEINFO *inf = VMALLOC(FILEINFO, strlen(r->name));
  memcpy(inf, r, sizeof(*r));
  strcpy(inf->name, r->name);
  inf->refCt = 0;
  return inf;
}
