/****************************************************************************/
/*                                                                          */
/* cdf2sun.c                                                                */
/*                                                                          */
/* Read ASCII data generated by ADBDUMP and produce SUN cm format           */
/*                                                                          */
/* A. Garzotto, May '94                                                     */
/*                                                                          */
/* Updated for Win32 by James Edward Lewis II, 20 September 2009            */
/*                                                                          */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <cstring>
#include <ctype.h>
#include <fcntl.h>

#define YES 1
#define NO  0
#define NIL 0

#ifndef O_BINARY
#define O_BINARY 0
#endif

#define S_ALARM     1
#define S_CHECKOFF  2
#define S_MONTHLY   2
#define S_CARRY     4
#define S_WEEKLY    4
#define S_TODO     16
#define S_EVENT    32
#define S_STUB     64
#define S_APPT    128

/****************************************************************************/

struct rep_desc
{
   char freq;
   int days;
   int month;
   char y1;
   char m1;
   char d1;
   char y2;
   char m2;
   char d2;
   char ndeleted;
   char *deleted;
};

typedef struct rep_desc *REPEAT;

struct rec_desc
{
   int num;                 /* number of this record */
   char reptype;            /* repeat type */
   char state;              /* record state */
   char *desc;              /* description */
   char *location;          /* location */
   char year;               /* start date; */
   char month;
   char day;
   int stime;               /* start time */
   int etime;               /* end time */
   int duration;            /* durations in days */
   char prio[2];            /* todo priority */
   int lead;                /* alarm lead time */
   char *note;              /* note */
   REPEAT repeat;           /* pointer to repeat description */
   struct rec_desc *next;   /* pointer to next element in list */
};

typedef struct rec_desc *RECORD;

/****************************************************************************/

FILE *fin = stdin;       /* CDF file */
FILE *fout = stdout;     /* Sun cm data file */

RECORD records = NIL;    /* list of data records */
int numrecords = 0;      /* number of records in database */
int notenum = 0;         /* number of current note */

int debug = NO;          /* debug mode */
char dateformat[80] = "d.m.yyyy"; /* date input format */
int retain_crlf = NO;    /* expect CR/LF in text fields */
int tquotes = YES;       /* expect quotes around text fields */
int nquotes = NO;        /* expect quotes around number fields */

char *monthnames[] =
{
   "Jan",
   "Feb",
   "Mar",
   "Apr",
   "May",
   "Jun",
   "Jul",
   "Aug",
   "Sep",
   "Oct",
   "Nov",
   "Dec"
};

/****************************************************************************/
/* allocate memory */

char *my_malloc(int size)
{
   char *p;

   p = (char *)malloc(size);
   if (!p)
   {
      fprintf(stderr, "Memory allocation problems.\nAborted!\n");
      exit(1);
   }
   return p;
}

/****************************************************************************/
/* create a new record struct */

RECORD new_rec()
{
   RECORD r;

   r = (RECORD)my_malloc(sizeof(struct rec_desc));
   r->reptype = 0;
   r->year = r->month = r->day = r->state = 0;
   r->stime = r->etime = r->duration = r->lead = 0;
   r->desc = r->location = r->note = NIL;
   r->next = records;
   records = r;
   r->num = numrecords++;
   r->repeat = NIL;
   return r;
}

/****************************************************************************/
/* create a new repeat struct */

REPEAT new_rep()
{
   REPEAT r;

   r = (REPEAT)my_malloc(sizeof(struct rep_desc));
   r->freq = r->y1 = r->m1 = r->d1 = r->y2 = r->m2 = r->d2 = '\0';
   r->ndeleted = '\0';
   r->deleted = NIL;
   r->days = r->month = 0;
   return r;
}

/****************************************************************************/
/* check if file format is ok */

void check_syntax(char *buf, char *p, char ch, char *msg)
{
   int i;

   if (*p == ch) return;
   if (!*p || ((p > buf) && !p[-1]))
      fprintf(stderr, "Syntax error %s: unexpected end of line!\n", msg);
   else if (ch != '?')
      fprintf(stderr, "Syntax error %s: `%c' expected!\n", msg, ch);
   else
      fprintf(stderr, "Syntax error %s: unexpected character!\n", msg);
   ch = *p;
   *p = '\0';
   fprintf(stderr, "%s<***HERE***>%c%s", buf, ch, &p[1]);
   exit(1);
}

/****************************************************************************/
/* read number from ascii string */

int get_number(char **p)
{
   int n;

   if (nquotes) check_syntax(*p, (*p)++, '\"', "before number");
   sscanf(*p, "%d", &n);
   while (**p && (**p >= '0') && (**p <= '9')) (*p)++;
   if (nquotes) check_syntax(*p, (*p)++, '\"', "after number");
   return n;
}

/****************************************************************************/
/* compare two strings ignoring case */

int my_strncasecmp(char *s1, char *s2, int n)
{
   while (*s1 && *s2 && n && (toupper(*s1) == toupper(*s2)))
   {
      s1++; s2++; n--;
   }
   if (n) return YES;
   return NO;
}

/****************************************************************************/
/* read a date */

void get_date(char **p, char *y, char *m, char *d)
{
   char *f = dateformat;
   char *s = *p;
   int i, nq = nquotes;

   nquotes = NO;
   while (*f)
   {
      switch (*f)
      {
      case 'd': while (*f == 'd') f++;
                *d = (char)(get_number(p) - 1);
                break;
      case 'm': i = 0;
                while (*f == 'm') { f++; i++; }
                if (i > 2)
                {
                   i = 0;
                   while ((i < 12) && my_strncasecmp(*p, monthnames[i], 3))
                      i++;
                   if (i >= 12)
                      check_syntax(s, *p, '?', "in month name");
                   else
                      *m = (char)i;
                   (*p) += 3;
                }
                else
                   *m = (char)(get_number(p) - 1);
                break;
      case 'y': while (*f == 'y') f++;
                i = get_number(p);
                if (i >= 100)
                   *y = (char)(i - 1900);
                else
                   *y = (char)i;
                break;
      default: check_syntax(s, *p, *f, "in 'date'");
               (*p)++; f++;
               break;
      }
   }
   nquotes = nq;
}

/****************************************************************************/
/* read a time field */

int get_time(char **p)
{
   int h, m;
   char *buf = *p;

   if (tquotes) check_syntax(buf, (*p)++, '\"', "before 'time'");
   sscanf(*p, "%d:%d", &h, &m);
   while (**p && (**p != '\"') && (tquotes || (**p != ','))) (*p)++;
   if (tquotes) check_syntax(buf, (*p)++, '\"', "after 'time'");
   return (60 * h + m);
}

/****************************************************************************/
/* read a text or note field */

void get_text(char **p, char **str)
{
   char *q, *qq, *qend;
   char *buf = *p;

   if (tquotes) check_syntax(buf, (*p)++, '\"', "before text field");

   q = *p;
   while (*q)
   {
      if (retain_crlf && (*q == '\n') && (q != *p))
         fgets(&q[1], 1024, fin);
      if (tquotes && (*q == '\"') && (q[-1] != '\\')) break;
      if (!tquotes && (*q == ',') && (q[-1] != '\\')) break;
      if (!tquotes && ((*q == '\n') || (*q == '\r'))) break;
      q++;
   }
   qend = q;

   *str = (char *)my_malloc(qend - *p + 2);
   q = *p;
   qq = *str;
   while (q < qend)
   {
      if (*q == '\\')
      {
         q++;
         switch (*q)
         {
         case 'r': *(qq++) = '\r'; break;
         case 'n': *(qq++) = '\n'; break;
         default: *(qq++) = *q;
         }
         q++;
      }
      else
         *(qq++) = *(q++);
   }
   *qq = '\0';
   *p = qend;
   if (tquotes) check_syntax(buf, (*p)++, '\"', "after text field");
}


/****************************************************************************/
/* add records from CDF file */

void load_new_records()
{
   char *buf;
   char *p;
   RECORD r;
   int num, i;

   buf = (char *)my_malloc(8194);
   while (fgets(buf, 8192, fin))
   {
      if (*buf > ' ')
      {
         r = new_rec();
         p = buf;
         if (tquotes) check_syntax(buf, p++, '\"', "(is this a CDF file?)");
         get_date(&p, &(r->year), &(r->month), &(r->day));
         if (tquotes) check_syntax(buf, p++, '\"', "after 'date'");
         check_syntax(buf, p++, ',', "after 'date'");
         get_text(&p, &(r->desc));
         check_syntax(buf, p++, ',', "after 'description'");
         get_text(&p, &(r->location));
         check_syntax(buf, p++, ',', "after 'location'");
         r->state = (char)get_number(&p);
         check_syntax(buf, p++, ',', "after 'state'");

         if (r->state & S_APPT)
         {
            r->stime = get_time(&p);
            check_syntax(buf, p++, ',', "after 'start time'");
            r->etime = get_time(&p);
            check_syntax(buf, p++, ',', "after 'end time'");
            r->duration = get_number(&p);
            check_syntax(buf, p++, ',', "after 'duration'");
            r->lead = get_number(&p);
            check_syntax(buf, p++, ',', "after 'lead time'");
         }
         else if (r->state & S_EVENT)
         {
            r->duration = get_number(&p);
            check_syntax(buf, p++, ',', "after 'duration'");
            r->stime = r->etime = -1;
            r->lead = 0;
         }
         else /* TODO */
         {
            if (tquotes) check_syntax(buf, p++, '\"', "before 'priority'");
            r->prio[0] = *(p++);
            r->prio[1] = *(p++);
            if (tquotes) check_syntax(buf, p++, '\"', "after 'priority'");
            check_syntax(buf, p++, ',', "after 'priority'");
            r->duration = get_number(&p);
            check_syntax(buf, p++, ',', "after 'duration'");
         }
         get_text(&p, &(r->note));
         if (*p == ',')
         {
            p++;
            r->reptype = (char)get_number(&p);
            check_syntax(buf, p++, ',', "after 'repeat type'");
            r->repeat = new_rep();
            r->repeat->freq = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->days = get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->month = get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->y1 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->m1 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->d1 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->y2 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->m2 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->d2 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->ndeleted = (char)get_number(&p);
            num = (int)r->repeat->ndeleted & 255;
            if (num)
            {
               check_syntax(buf, p++, ',', "");
               r->repeat->deleted = (char *)my_malloc(4 * num);
               for (i = 0; i < 4 * num; i++)
               {
                  check_syntax(buf, p - 1, ',', "in 'deleted record list'");
                  r->repeat->deleted[i] = (char)get_number(&p);
                  p++;
               }
            }
         }
         else
            r->reptype = (char)1;
      }
   }
   fclose(fin);
   free(buf);
}

/****************************************************************************/
/* create a Sun cm file header */

void put_header()
{
   fprintf(fout, "Version: 1\n");
}

/****************************************************************************/
/* convert character set from IBM to ISO-8859-1 and handle special chars*/

void put_string(char *s)
{
   char *p = s;
   int lines = 1;

   while(*p && (lines < 4))
   {
      if (((int)*p & 255) > 127)
      {
         switch (*p)
         {
         case '\204': fprintf(fout, "\344"); break;
         case '\224': fprintf(fout, "\366"); break;
         case '\201': fprintf(fout, "\374"); break;
         case '\216': fprintf(fout, "\304"); break;
         case '\231': fprintf(fout, "\326"); break;
         case '\232': fprintf(fout, "\334"); break;
         case '\341': fprintf(fout, "\337");; break;
         case '\202': fprintf(fout, "\351");; break;
         default: fprintf(fout, "~");
                  fprintf(stderr,
                     "WARNING: replaced unknown character (%d) in '%s'\n",
                     (int)*p & 255, s);
         }
      }
      else
      {
         if (*p == '\n')
         {
            fprintf(fout, "\\n");
            lines++;
         }
         else if (*p == '\"')
            fprintf(fout, "\\\"");
         else if (*p != '\"')
            fprintf(fout, "%c", *p);
      }
      p++;
   }
  if (lines >= 4)
  {
      s[40] = '\0';
      fprintf(stderr, "WARNING: truncated '%s...'\n", s);
  }
}

/****************************************************************************/
/* repeat warning */

void sorry(RECORD r)
{
   fprintf(stderr, "WARNING: cannot handle repeat state of '%s' correctly!\n",
           r->desc);
}

/****************************************************************************/
/* write a data record to the .ADB file */

void write_record(RECORD r)
{
   int h;
   static int key = 1;

   fprintf(fout, "(add \"%s %d", monthnames[r->month], r->day + 1);
   if (r->stime < 0) r->stime = 0;
   if (r->etime < 0) r->etime = 1;
   fprintf(fout, " %d:%.2d:00 %d\"", r->stime / 60, r->stime % 60, r->year + 1900);
   fprintf(fout, " key: %d", key++);
   fprintf(fout, " what: \"");
   put_string(r->desc);
   fprintf(fout, "\\n");
   if (r->location && *(r->location))
   {
      fprintf(fout, "@");
      put_string(r->location);
      fprintf(fout, "\\n");
   }
   if (r->note && *(r->note))
   {
      put_string(r->note);
      fprintf(fout, "\\n");
   }
   fprintf(fout, "\"");
   fprintf(fout, " details: \"%s\"", ""); /* ??? */
   h = r->etime - r->stime;
   if (h < 0) h += 24 * 60;
   fprintf(fout, " duration: %d", 60 * h);
   fprintf(fout, " period: ");
   if (!(r->state & S_TODO) && (r->duration > 0))
   {
      fprintf(fout, " daily ntimes: %d", r->duration + 1);
   }
   else
   {
      switch((int)r->reptype)
      {
      case 2: fprintf(fout, "daily ntimes: 31"); sorry(r); break;
      case 4: fprintf(fout, "weekly ntimes: 52"); sorry(r); break;
      case 8: fprintf(fout, "monthly ntimes: 12"); sorry(r); break;
      case 16: fprintf(fout, "yearly ntimes: 100"); sorry(r); break;
      case 32: fprintf(fout, "yearly ntimes: 100"); sorry(r); break;
      default: fprintf(fout, "single ntimes: 0"); break;
      }
   }
   fprintf(fout, " author: \"%s@%s\"", getenv("USER"), getenv("HOST"));
   if (r->state & S_ALARM)
      fprintf(fout, " attributes: ((\"op\",\"%d\",\"\")(\"bp\",\"%d\",\"\"))",
              r->lead * 60, r->lead * 60);
   if (r->state & S_TODO)
      fprintf(fout, " tags: ((toDo, 0))");
   else if (r->state & S_EVENT)
      fprintf(fout, " tags: ((appointment, 0))");
   else
      fprintf(fout, " tags: ((appointment, 1))");
   fprintf(fout, " apptstat: active");
   fprintf(fout, " privacy: public");
   fprintf(fout, ")\n");

}

/****************************************************************************/
/* add new records to .ADB file */

void write_new_records()
{
   RECORD r = records;

   while (r)
   {
      write_record(r);
      r = r->next;
   }
   fclose(fout);
}

/****************************************************************************/
/* display usage information */

void help(char *prog)
{
   fprintf(stderr, "CDF2SUN version 1.0 by Andreas Garzotto\n\n");
   fprintf(stderr, "Update for Win32 by James Edward Lewis II\n\n");
   fprintf(stderr,
      "USAGE: %s [options] [<CDF file> [<Sun cm file>]] \n", prog);
   fprintf(stderr, " Options:\n");
   fprintf(stderr,
      "  -d fmt expect date as specified in fmt (default: 'd.m.yyyy')\n");
   fprintf(stderr, "  -q0    do not expect quotes around any fields\n");
   fprintf(stderr,
      "  -q1    expect quotes around text fields only (default)\n");
   fprintf(stderr,
      "  -q2    expect quotes around text fields and numbers\n");
   fprintf(stderr,
      "  -r     expect CR/LF in note fields (not '\\r\\n')\n");

   exit(1);
}

/****************************************************************************/
/* decode options */

void get_options(int argc, char **argv)
{
   int state = 0, i = 1;

   while (i < argc)
   {
      if (argv[i][0] == '-')
      {
         switch (argv[i][1])
         {
         case 'x': debug = YES; break;
         case 'd': if (i >= argc - 1) break;
                   strcpy(dateformat, argv[++i]);
                   break;
         case 'q': if (argv[i][2] == '0') tquotes = NO;
                   if (argv[i][2] == '2') nquotes = YES;
                   break;
         case 'r': retain_crlf = YES; break;
         default: help(argv[0]);
         }
      }
      else
      {
         if (state == 0)
         {
            fin = fopen(argv[i], "r");
            if (fin == NULL)
            {
               fprintf(stderr, "Cannot open input file '%s'\n", argv[i]);
               exit(1);
            }
            state = 1;
         }
         else if (state == 1)
         {
            fout = fopen(argv[i], "w");
            if (fout == NULL)
            {
               fprintf(stderr, "Cannot open output file '%s'\n", argv[i]);
               exit(1);
            }
            state = 2;
         }
         else
            help(argv[0]);
      }
      i++;
   }
}

/****************************************************************************/

int main(int argc, char **argv)
{
   get_options(argc, argv);
   load_new_records();
   put_header();
   write_new_records();
   return 0;
}

/****************************************************************************/
