/* LUNITS.C -- Units Commands

	Written April 1994 by Craig A. Finseth
	Copyright 1994 by Craig A. Finseth
*/

#include "loki.h"
#if defined(FLOAT)
#include <math.h>
#endif

static char u_ascii_buf[SMALLBUFFSIZE];

char *U_ASCII(int chr);

/* ------------------------------------------------------------ */

/* Read a string into a number, assume that it's type TYPE.  Numbers
are of these forms:

	<part1>		Enter number assuming it is in supplied type.

	<part1><unit1>	Enter number according to type.  If <unit1> is
			a multi-part unit, multiply by all parts.

	<part1><unit1><part2>	Enter number according to type.  If
			<unit1> is a multi-part unit, multiply by all
			parts but the first.

	<part1><unit1><part2><unit2>	Same as previous.

	<part1><unit1><part2><unit2><part3>	Enter number according
			to type.  If <unit1> is a multi-part unit,
			multiply by all parts but the first.

	<part1><unit1><part2><unit2><part3><unit3>	Same as previous.

	<part1><unit1><part2><unit2><part3><unit3><part4>	Same
			as previous, but one more thing to add
			(usually a decimal part only).

Each <part> can be of the form implied by this regular expression:

	[+-][0-9.,]*[eE[+-]][0-9]*

(Actually, commas are converted to decimal points early on for
simplicity.)

This breaks out to:

	- leading + or - sign
	- digits 0-9 and 0, 1, or 2 decimal points
	- an optional E followed by an optional + or - sign
	- optional exponent digits (default 0)

If there are 0 or 1 decimal points, the number is a normal number.
With 2 decimal points, the number is a fraction.  Interpretations:

	a . b . c	==>	a b / c

Defaults:

	  .   .  	zero
	  .   . c	zero (0 + 0 / c)
	  . b .  	b / denominator
	  . b . c	b / c
	a .   .  	a
	a .   . c	a / c
	a . b .  	a + b / denominator (differs here from HP32SII)

But wait, there's more!  What about the lower case letters a-f?  There
are two cases:

	1) Hex letters mode is on (by either m.hexletters is true or the
	current unit is hex, zerohex, or ascii).

	2) Hex letters mode is off.

If the mode is on, then the lowercase letters a-f will be assumed to
imply hex letters.  Thus, you can have a current unit of octal and type:

	12cfh

and the number will be assumed to be 12cf hexadecimal (and not a
typo).  If you wanted to enter cubic feet, you can do:

	12CuF

If hex letters mode is off, you can still enter a hexadecimal number
assuming that it only has the digits 0-9.

Whew!
*/

FLAG
UEnter(struct number *a, char *str, int type)
	{
	char buf[LINEBUFFSIZE];
	char part1[SMALLBUFFSIZE];
	char part2[SMALLBUFFSIZE];
	char part3[SMALLBUFFSIZE];
	char part4[SMALLBUFFSIZE];
	char unit1[SMALLBUFFSIZE];
	char unit2[SMALLBUFFSIZE];
	char unit3[SMALLBUFFSIZE];
	char smallbuf[UNIT_WIDTH];
	char *cptr;
	char *savecptr;
	char *numstr = Res_String(NULL, RES_MSGS,
		(m.hex_letters || m.cur_type == (RES_UN_HEX / UNIT_SIZE) ||
		 m.cur_type == (RES_UN_ASCII / UNIT_SIZE) ||
		 m.cur_type == (RES_UN_HEXZ / UNIT_SIZE)) ?
		 RES_HEXSTR : RES_NUMBERSTR); 
	char echr = Res_Char(RES_MSGS, RES_ESTR); 
	int amt;
	int cnt;
	struct number p1;
	struct number p2;
	struct number p3;
	struct number p4;
#if defined(BCD)
	struct number tmp;
#endif

/* make working copy */
	xstrncpy(buf, str);

/* get rid of commas */
	for (cptr = buf; *cptr != NUL; cptr++) {
		if (*cptr == ',') *cptr = '.';
		}

/* get part 1 */
	savecptr = buf;
	for (cptr = buf; *cptr != NUL &&
		(*sindex(numstr, *cptr) != NUL || ConvUpper(*cptr) == echr);
		cptr++) ;
	amt = min(cptr - savecptr, sizeof(part1) - 1);
	memmove(part1, savecptr, amt);
	part1[amt] = NUL;

/* get unit 1 */
	savecptr = cptr;
	for ( ; *cptr != NUL && *sindex(numstr, *cptr) == NUL; cptr++) ;
	amt = min(cptr - savecptr, sizeof(unit1) - 1);
	memmove(unit1, savecptr, amt);
	unit1[amt] = NUL;

/* get part 2 */
	savecptr = cptr;
	for ( ; *cptr != NUL &&
		(*sindex(numstr, *cptr) != NUL || xtoupper(*cptr) == echr);
		cptr++) ;
	amt = min(cptr - savecptr, sizeof(part2) - 1);
	memmove(part2, savecptr, amt);
	part2[amt] = NUL;

/* get unit 2 */
	savecptr = cptr;
	for ( ; *cptr != NUL && *sindex(numstr, *cptr) == NUL; cptr++) ;
	amt = min(cptr - savecptr, sizeof(unit2) - 1);
	memmove(unit2, savecptr, amt);
	unit2[amt] = NUL;

/* get part 3 */
	savecptr = cptr;
	for ( ; *cptr != NUL &&
		(*sindex(numstr, *cptr) != NUL || xtoupper(*cptr) == echr);
		cptr++) ;
	amt = min(cptr - savecptr, sizeof(part3) - 1);
	memmove(part3, savecptr, amt);
	part3[amt] = NUL;

/* get unit 3 */
	savecptr = cptr;
	for ( ; *cptr != NUL && *sindex(numstr, *cptr) == NUL; cptr++) ;
	amt = min(cptr - savecptr, sizeof(unit3) - 1);
	memmove(unit3, savecptr, amt);
	unit3[amt] = NUL;

/* get part 4 */
	savecptr = cptr;
	for ( ; *cptr != NUL &&
		(*sindex(numstr, *cptr) != NUL || xtoupper(*cptr) == echr);
		cptr++) ;
	amt = min(cptr - savecptr, sizeof(part4) - 1);
	memmove(part4, savecptr, amt);
	part4[amt] = NUL;

/* let's get lazy and ignore the rest */

/* no unit? only number, so return it */
	if (unit1[0] == NUL) {
		return(UEnterNum(a, part1, type));
		}

/* there's a unit, so let's find it */

	for (cnt = num_units - 1; cnt >= 0; cnt--) {
		xstrcpy(smallbuf, units[cnt].name);
		*sindex(smallbuf, SP) = NUL;
		if (strequ(unit1, smallbuf)) {
			break;
			}
		}
	if (cnt < 0) {
		DError(RES_BADUNIT);
		return(FALSE);
		}

/* have valid unit, so let's handle the part2, part3, and part4 */

	cptr = units[cnt].factors;
	amt = units[cnt].len;

/* no part2/part3 */
	if (amt <= 1) {
		return(UEnterNum(a, part1, cnt));
		}

/* 2 or 3 parts */

	if (!UEnterNum(&p1, part1, RES_UN_NONE / UNIT_SIZE)) {
		DError(RES_BADNUM);
		return(FALSE);
		}
	if (*part2 == NUL) {
#if defined(FLOAT)
		p2.f = 0.0;
#endif
#if defined(BCD)
		BAssign(&p2, 0);
#endif
		}
	else if (!UEnterNum(&p2, part2, RES_UN_NONE / UNIT_SIZE)) {
		DError(RES_BADNUM);
		return(FALSE);
		}

	if (*part3 == NUL) {
#if defined(FLOAT)
		p3.f = 0.0;
#endif
#if defined(BCD)
		BAssign(&p3, 0);
#endif
		}
	else if (!UEnterNum(&p3, part3, RES_UN_NONE / UNIT_SIZE)) {
		DError(RES_BADNUM);
		return(FALSE);
		}

	if (*part4 == NUL) {
#if defined(FLOAT)
		p4.f = 0.0;
#endif
#if defined(BCD)
		BAssign(&p4, 0);
#endif
		}
	else if (!UEnterNum(&p4, part4, RES_UN_NONE / UNIT_SIZE)) {
		DError(RES_BADNUM);
		return(FALSE);
		}

	if (amt == 2) {
#if defined(FLOAT)
		a->f = p1.f * (*cptr) + p2.f + p3.f;
#endif
#if defined(BCD)
		BAssignI(a, (int)*cptr);
		BMul(a, &p1);
		BAdd(a, &p2);
		BAdd(a, &p3);
#endif
		}
	else	{
#if defined(FLOAT)
		a->f = (p1.f * (*cptr) + p2.f) * (*(cptr + 1)) + p3.f + p4.f;
#endif
#if defined(BCD)
		BAssignI(a, (int)*cptr);
		BMul(a, &p1);
		BAdd(a, &p2);

		BAssignI(&tmp, (int)*(cptr + 1));
		BMul(a, &tmp);
		BAdd(a, &p3);
		BAdd(a, &p4);
#endif
		}
	a->type = cnt;
	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Remember the pesky fractions and actual number entry from the
above? Well, do one number. */

FLAG
UEnterNum(struct number *a, char *str, int type)
	{
	char buf[LINEBUFFSIZE];
	char *cptr;
	char *savecptr;
	int cnt;
	int n1;
	int n2;
	int n3;
	int dots;
	FLAG wasneg;
#if defined(BCD)
	struct number tmp;
#endif

	a->type = type;

/* make working copy */
	xstrncpy(buf, str);

/* get rid of commas, count .s */
	dots = 0;
	for (cptr = buf; *cptr != NUL; cptr++) {
		if (*cptr == ',') *cptr = '.';
		if (*cptr == '.') dots++;
		}

/* if 0 or 1, regular number */
	if (dots < 2) {
		if (type == (RES_UN_BIN / UNIT_SIZE) ||
			 type == (RES_UN_BINZ / UNIT_SIZE)) {
			if (!MSToNL(buf, &a->i, 2)) {
				DError(RES_BADNUM);
				return(FALSE);
				}
			a->i &= bin_mask;
			return(TRUE);
			}
		else if (type == (RES_UN_OCT / UNIT_SIZE) ||
			 type == (RES_UN_OCTZ / UNIT_SIZE)) {
			if (!MSToNL(buf, &a->i, 8)) {
				DError(RES_BADNUM);
				return(FALSE);
				}
			a->i &= bin_mask;
			return(TRUE);
			}
		else if (type == (RES_UN_DEC / UNIT_SIZE)) {
			if (!MSToNL(buf, &a->i, 10)) {
				DError(RES_BADNUM);
				return(FALSE);
				}
			a->i &= bin_mask;
			return(TRUE);
			}
		else if (type == (RES_UN_HEX / UNIT_SIZE) ||
			 type == (RES_UN_HEXZ / UNIT_SIZE) ||
			 type == (RES_UN_ASCII / UNIT_SIZE)) {
			if (!MSToNL(buf, &a->i, 16)) {
				DError(RES_BADNUM);
				return(FALSE);
				}
			a->i &= bin_mask;
			return(TRUE);
			}
		else	{
			/* handle leading E */
			cptr = buf;
			if (*cptr == '-') cptr++;
			if (ConvUpper(*cptr) == Res_Char(RES_MSGS, RES_ESTR)) {
				memmove(cptr + 1, cptr, strlen(cptr) + 1);
				*cptr = '1';
				}
#if defined(FLOAT)
			if (sscanf(buf, "%lf", &a->f) != 1) {
#endif
#if defined(BCD)
			if (!BEnter(a, buf)) {
#endif
				DError(RES_BADNUM);
				return(FALSE);
				}
			a->type = type;
			return(TRUE);
			}
		}

/* if >2, error */
	if (dots > 2) {
		DError(RES_BADNUM);
		return(FALSE);
		}

/* fractions */
	if (units[type].cat == 1) {
		/* fractions, on a binary number? */
		DError(RES_BADNUM);
		return(FALSE);
		}	

	n1 = 0;
	n2 = 0;
	n3 = 0;

/* get part 1 */
	savecptr = buf;
	for (cptr = buf; *cptr != NUL && *cptr != '.'; cptr++) ;
	*cptr++ = NUL;
	if (!SToN(savecptr, &n1, 10)) {
		DError(RES_BADNUM);
		return(FALSE);
		}	

/* get part 2 */
	savecptr = cptr;
	for ( ; *cptr != NUL && *cptr != '.'; cptr++) ;
	*cptr++ = NUL;
	if (!SToN(savecptr, &n2, 10)) {
		DError(RES_BADNUM);
		return(FALSE);
		}	

/* get part 3 */
	if (!SToN(cptr, &n3, 10)) {
		DError(RES_BADNUM);
		return(FALSE);
		}

	if (n1 < 0) {
		wasneg = TRUE;
		n1 = -n1;
		}
	else	{
		wasneg = FALSE;
		}
	if (n2 != 0) {
		if (n3 != 0) {
#if defined(FLOAT)
			a->f = (double)n1 + ((double)(n2)) / ((double)(n3));
#endif
#if defined(BCD)
			BAssignI(a, n2);
			BAssignI(&tmp, n3);
			BDiv(a, &tmp);
			BAssignI(&tmp, n1);
			BAdd(a, &tmp);
#endif
			}
		else	{
#if defined(FLOAT)
			a->f = (double)n1 + ((double)(n2)) /
				((double)(m.frac_denom));
#endif
#if defined(BCD)
			BAssignI(a, n2);
			BAssignI(&tmp, m.frac_denom);
			BDiv(a, &tmp);
			BAssignI(&tmp, n1);
			BAdd(a, &tmp);
#endif
			}
		}
	else	{
		if (n3 != 0) {
#if defined(FLOAT)
			a->f = ((double)(n1)) / ((double)(n3));
#endif
#if defined(BCD)
			BAssignI(a, n1);
			BAssignI(&tmp, n3);
			BDiv(a, &tmp);
#endif
			}
		else	{
#if defined(FLOAT)
			a->f = (double)n1;
#endif
#if defined(BCD)
			BAssignI(a, n1);
#endif
			}
		}
#if defined(FLOAT)
	if (wasneg) a->f = -a->f;
#endif
#if defined(BCD)
	if (wasneg) BNeg(a);
#endif
	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Format the supplied number into the supplied buffer according to
the specified type. */

void
UFormat(char *buf, int type, struct number *nptr)
	{
	struct number n = *nptr;
	char *fmt = units[type].format;
	char buf2[UNIT_WIDTH];
	int wid;
	int len;
	char *factors;
	long num;
	long n1;
	long n2;
	long n3;
	FLAG wasneg;
#if defined(FLOAT)
	double ftmp;
#endif
#if defined(BCD)
	struct number ftmp;
#endif

	UToType(type, &n);

	switch (*fmt) {

	case SP:		/* xprintf string */
		if (type == (RES_UN_BINZ / UNIT_SIZE)) {
			wid = m.word_size;
			lxsprintf(buf, fmt + 1, (long)wid, n.i);
			}
		else if (type == (RES_UN_OCTZ / UNIT_SIZE)) {
			wid = (m.word_size + 2) / 3;
			lxsprintf(buf, fmt + 1, (long)wid, n.i);
			}
		else if (type == (RES_UN_HEXZ / UNIT_SIZE)) {
			wid = (m.word_size + 3) / 4;
			lxsprintf(buf, fmt + 1, (long)wid, n.i);
			}
		else if (type == (RES_UN_DEC / UNIT_SIZE)) {
			if (m.compl_mode == 0) {
				lxsprintf(buf, "%ud", n.i);
				}
			else if (m.compl_mode == 1) {
				if (m.word_size <= BIN_SIZE &&
					 (n.i & (1L << (m.word_size - 1)))) {
					n.i |= (~0L) << m.word_size;
					}
				if (n.i < 0) n.i++;
				lxsprintf(buf, "%dd", n.i);
				}
			else	{
				if (m.word_size <= BIN_SIZE &&
					 (n.i & (1L << (m.word_size - 1)))) {
					n.i |= (~0L) << m.word_size;
					}
				lxsprintf(buf, "%dd", n.i);
				}
			}
		else	{
			lxsprintf(buf, fmt + 1, n.i);
			}
		break;

	case 'G':		/* float */
		if (type == (RES_UN_NONE / UNIT_SIZE)) {
			if (m.frac_mode != 0) {
#if defined(FLOAT)
				n.f = modf(n.f, &ftmp);
				sprintf(buf, "%.9lg%s", ftmp, FFormat(&n));
#endif
#if defined(BCD)
				BModf(&n, &ftmp);
				BFormat(buf, &n);
				strcat(buf, FFormat(&ftmp));
#endif
				}
			else	{
#if defined(FLOAT)
				sprintf(buf, "%.9lg", n.f);
#endif
#if defined(BCD)
				BFormat(buf, &n);
#endif
				}
			return;
			}

		factors = units[type].factors;
		len = units[type].len;

		if (len <= 1) {
			if (m.frac_mode != 0) {
#if defined(FLOAT)
				n.f = modf(n.f, &ftmp);
				sprintf(buf, "%.9lg%s%s",
					ftmp,
					FFormat(&n),
					units[type].name);
#endif
#if defined(BCD)
				BModf(&n, &ftmp);
				BFormat(buf, &n);
				strcat(buf, FFormat(&ftmp));
				strcat(buf, units[type].name);
#endif
				}
			else	{
#if defined(FLOAT)
				sprintf(buf, "%.9lg%s",
					n.f,
					units[type].name);
#endif
#if defined(BCD)
				BFormat(buf, &n);
				strcat(buf, units[type].name);
#endif
				}
			return;
			}

#if defined(FLOAT)
		if (n.f < 0) {
			wasneg = TRUE;
			n.f = -n.f;
			}
		else	{
			wasneg = FALSE;
			}
		num = n.f;
		n.f = modf(n.f, &ftmp);
#endif
#if defined(BCD)
		if (BIsNeg(&n)) {
			wasneg = TRUE;
			BNeg(&n);
			}
		else	{
			wasneg = FALSE;
			}
		num = BLong(&n);
		BModf(&n, &ftmp);
#endif

		if (len == 2) {
			n1 = num / *factors;
			n2 = num % *factors;
			n3 = 0;
			}
		else	{
			n1 = num / (*factors * *(factors + 1));
			num %= (*factors * *(factors + 1));
			n2 = num / *(factors + 1);
			n3 = num % *(factors + 1);
			}
		if (wasneg) *buf++ = '-';
		lxsprintf(buf, fmt + 1, n1, n2, n3);
		buf += strlen(buf);
#if defined(FLOAT)
		if (m.frac_mode != 0) {
			xstrcpy(buf2, FFormat(&n));
			}
		else	{
			sprintf(buf2, "%.9lg", n.f);
#endif
#if defined(BCD)
		if (m.frac_mode != 0) {
			xstrcpy(buf2, FFormat(&ftmp));
			}
		else	{
			BFormat(buf2, &ftmp);
#endif
			}
		xstrcpy(buf, buf2 + ((*buf2 == '0') ? 1 : 0));
		break;

	case 'K':		/* key */
		xstrcpy(buf, U_ASCII(n.i));
		break;
		}
	
	}

/* ------------------------------------------------------------ */

/* Return TRUE if the type is a binary one. */

FLAG
UIsBin(int type)
	{
	return(units[type].cat == 1);
	}


/* ------------------------------------------------------------ */

/* Promote the specified number by AMT, which may be postive or
negative. */

void
UPromote1(struct number *a, int amt)
	{
	int cnt;
	int x;
	int new;

/* handle alternates */
	for (cnt = Res_Number(RES_ALTERNATES, -1) - 3; cnt >= 0; cnt -= 2) {
		x = Res_Number(RES_ALTERNATES, cnt) / UNIT_SIZE;
		if (a->type == x) {
			UToType(Res_Number(RES_ALTERNATES, cnt + 1) /
				UNIT_SIZE, a);
			break;
			}
		}

/* find in promote */
	for (cnt = Res_Number(RES_PROMOTE, -1) - 1; cnt >= 0; cnt--) {
		x = Res_Number(RES_PROMOTE, cnt) / UNIT_SIZE;
		if (a->type == x) break;
		}

/* not promotable: punt */
	if (cnt < 0) return;

	new = (cnt % 3) + amt;
	new = max(0, min(new, 2));
	a->type = Res_Number(RES_PROMOTE, (cnt / 3) * 3 + new) / UNIT_SIZE;
	}


/* ------------------------------------------------------------ */

/* Adjust the units of the two numbers to be in sync, then promote the
first by the unit of the second.  If AMT is +1, add them. If AMT is
-1, subtract the second from the first. */

void
UPromote2(struct number *a, struct number *b, int amt)
	{
	int cnta;
	int xa;
	int cntb;
	int xb;
	int new;

/* handle alternates for a */
	for (cnta = Res_Number(RES_ALTERNATES, -1) - 3; cnta >= 0; cnta -= 2) {
		xa = Res_Number(RES_ALTERNATES, cnta) / UNIT_SIZE;
		if (a->type == xa) {
			UToType(Res_Number(RES_ALTERNATES, cnta + 1) /
				UNIT_SIZE, a);
			break;
			}
		}

/* find a in promote */
	for (cnta = Res_Number(RES_PROMOTE, -1) - 1; cnta >= 0; cnta--) {
		xa = Res_Number(RES_PROMOTE, cnta) / UNIT_SIZE;
		if (a->type == xa) break;
		}

/* is it promotable? */
	if (cnta < 0) return;

/* handle alternates for b */
	for (cntb = Res_Number(RES_ALTERNATES, -1) - 3; cntb >= 0; cntb -= 2) {
		xb = Res_Number(RES_ALTERNATES, cntb) / UNIT_SIZE;
		if (b->type == xb) {
			UToType(Res_Number(RES_ALTERNATES, cntb + 1) /
				UNIT_SIZE, b);
			break;
			}
		}

/* find b in promote */
	for (cntb = Res_Number(RES_PROMOTE, -1) - 1; cntb >= 0; cntb--) {
		xb = Res_Number(RES_PROMOTE, cntb) / UNIT_SIZE;
		if (b->type == xb) break;
		}

/* is b promotable? */
	if (cntb < 0) {
		/* no, so just add or subtract 1 */
		new = (cnta % 3) + amt;
		new = max(0, min(new, 2));
		a->type = Res_Number(RES_PROMOTE, (cnta / 3) * 3 + new) /
			UNIT_SIZE;
		return;
		}

/* yes, now convert it to the corresponding power of a */
	new = cntb % 3;
	UToType(Res_Number(RES_PROMOTE, (cnta / 3) * 3 + new) / UNIT_SIZE, b);

/* ok, so now figure the new power of a */
	new = (cnta % 3) + ((cntb % 3) + 1) * amt;
	new = max(0, min(new, 2));
	a->type = Res_Number(RES_PROMOTE, (cnta / 3) * 3 + new) / UNIT_SIZE;
	}


/* ------------------------------------------------------------ */

/* Convert the number to a binary type.  If it is one, do nothing.
Otherwise, if the default type is a binary type, convert to that.
Otherwise, convert to hex. */

void
UToBin(struct number *a)
	{
	if (units[a->type].cat == 1) return;
	UToType(units[m.cur_type].cat == 1 ?
		m.cur_type : RES_UN_HEX / UNIT_SIZE, a);
	}


/* ------------------------------------------------------------ */

/* Convert the number to the specified type. */

void
UToType(int type, struct number *a)
	{
	int typecat = units[type].cat;
	int acat = units[a->type].cat;
	struct number tmp;
	int num;
	int cnt;
	char *how;

/* no conversion needed */

	if (a->type == type) return;

/* incompatible types, punt conversion */

	if (typecat != acat) {
		if (typecat == 1) {
#if defined(FLOAT)
			a->i = a->f;
#endif
#if defined(BCD)
			a->i = BLong(a);
#endif
			a->i &= bin_mask;
			}
		else if (acat == 1) {
#if defined(FLOAT)
			a->f = a->i;
#endif
#if defined(BCD)
			BAssignL(a, a->i);
#endif
			}
		a->type = type;
		return;
		}		

/* compatible types (same category), so convert */

	num = Res_Number(RES_CONVERT, -1) - 1;
	for (cnt = 0; cnt < num; cnt += 3) {
		if (a->type == (Res_Number(RES_CONVERT, cnt) / UNIT_SIZE) &&
			 type == (Res_Number(RES_CONVERT, cnt + 1) /
			 	UNIT_SIZE)) {
			break;
			}
		}

	/* found a conversion */
	if (cnt < num) {
		how = Res_String(NULL, RES_CONVERT, cnt + 2);
		switch (*how) {

		case '*':	/* multiply by value */
			UEnterNum(&tmp, how + 1, type);
#if defined(FLOAT)
			a->f *= tmp.f;
#endif
#if defined(BCD)
			BMul(a, &tmp);
#endif
			break;

		case '1':	/* do deg C to deg F */
#if defined(FLOAT)
			a->f = a->f * 1.8 + 32;
#endif
#if defined(BCD)
			BAssign(&tmp, 9);
			BMul(a, &tmp);
			BAssign(&tmp, 5);
			BDiv(a, &tmp);
			BAssignI(&tmp, 32);
			BAdd(a, &tmp);
#endif
			break;

		case '2':	/* do deg C to K */
#if defined(FLOAT)
			a->f += 273.15;
#endif
#if defined(BCD)
			BEnter(&tmp, "273.15");
			BAdd(a, &tmp);
#endif
			break;

		case '3':	/* do deg F to deg C */
#if defined(FLOAT)
			a->f = (a->f - 32) / 1.8;
#endif
#if defined(BCD)
			BAssignI(&tmp, 32);
			BSub(a, &tmp);
			BAssign(&tmp, 5);
			BMul(a, &tmp);
			BAssign(&tmp, 9);
			BDiv(a, &tmp);
#endif
			break;

		case '4':	/* do deg F to K */
#if defined(FLOAT)
			a->f = (a->f - 32) / 1.8 + 273.15;
#endif
#if defined(BCD)
			BAssignI(&tmp, 32);
			BSub(a, &tmp);
			BAssign(&tmp, 5);
			BMul(a, &tmp);
			BAssign(&tmp, 9);
			BDiv(a, &tmp);
			BEnter(&tmp, "273.15");
			BAdd(a, &tmp);
#endif
			break;

		case '5':	/* do K to deg C */
#if defined(FLOAT)
			a->f -= 273.15;
#endif
#if defined(BCD)
			BEnter(&tmp, "273.15");
			BSub(a, &tmp);
#endif
			break;

		case '6':	/* do K to deg F */
#if defined(FLOAT)
			a->f = (a->f - 273.15) * 1.8 + 32;
#endif
#if defined(BCD)
			BEnter(&tmp, "273.15");
			BSub(a, &tmp);
			BAssign(&tmp, 9);
			BMul(a, &tmp);
			BAssign(&tmp, 5);
			BDiv(a, &tmp);
			BAssignI(&tmp, 32);
			BAdd(a, &tmp);
#endif
			break;
			}
		}
	a->type = type;
	}


/* ------------------------------------------------------------ */

/* Return a pointer to a static buffer that contains the carat
notation form of the supplied character. */

char *
U_ASCII(int chr)
	{
	char *cptr;

	if (chr < 256) {
		return(TPrintChar(chr));
		}
	else if (chr < 512) {
		chr &= 0xff;
		cptr = Res_String(NULL, RES_KEYLABEL, chr);
		if (*cptr != NUL) {
			return(cptr);
			}
		xsprintf(u_ascii_buf, "??%02x", chr);
		return(u_ascii_buf);
		}
	else	{
		xsprintf(u_ascii_buf, "??%04x", chr);
		return(u_ascii_buf);
		}
	}


/* end LUNITS.C -- Units Commands */
