/* LMISC.C -- Miscellaneous Commands

	Written July 1991 by Craig A. Finseth
	Copyright 1994 by Craig A. Finseth
*/

#include "loki.h"
#if defined(MSDOS)
#include <time.h>
#endif

static int mconv_uarg = -1;

/* ------------------------------------------------------------ */

/* Delete the last entered character.  If no entered character, clear X. */

void
CCharBD(void)
	{
	int x = strlen(m.line);

	if (x > 0) {
		m.line[x - 1] = NUL;
		}
	else	{
		AZero(&m.r[X]);
		}
	DChanged(US_ENTRY);
	}


/* ------------------------------------------------------------ */

/* If lower case, convert to upper case. */

int
ConvUpper(int chr)
	{
	if (chr < 0 || chr > 255) return(chr);
	return((chr + *(unsigned char *)(upper + chr)) & 0xFF);
	}


/* ------------------------------------------------------------ */

/* Abort the current command prefix. */

void
MAbort(void)
	{
	TBell();
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Do the About command. */

void
MAbout(void)
	{
	char buf[BUFFSIZE];
	FLAG wastext = xprintf_get_text();

	xprintf_set_text(FALSE);
	xsprintf(buf, Res_String(NULL, RES_MSGS, RES_ABOUTSTR),
		madefor,
		key_method,
		screen_type,
		Res_Char(RES_CONF, RES_SCRNMETHOD),
		TMaxRow(),
		TMaxCol());
	DShow('T', buf, FALSE);
	TGetKey();
	xprintf_set_text(wastext);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* View all forms of the value. */

void
MAll(void)
	{
	char buf[COLMAX * 2];
	char smallbuf[UNIT_WIDTH];
	int cnt;
	int cnt2;

	do	{
		cnt2 = 0;
		for (cnt = num_units - 1; cnt >= 0; cnt--) {
			UFormat(smallbuf, cnt, &x);
			xsprintf(buf, "%8s     %s",
				units[cnt].name,
				smallbuf);
			DLine2(buf, cnt2++, TAttrNorm());

			if (cnt2 >= TMaxRow() - 3) {
				TSetPoint(cnt2, 0);
				DRowChar(RES_OTHERSPLITCHAR);
				TSetPoint(cnt2 + 1, 0);
				TForce();
				KGetChar();
				cnt2 = 0;
				}
			}
		DClear(0, TMaxRow() - 3);
		TSetPoint(cnt2, 0);
		DRowChar(RES_OTHERSPLITCHAR);
		TSetPoint(cnt2 + 1, 0);
		TForce();
		} while (KGetChar() == KEYREGEN);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Fix the bin mask. */

void
MBinFix(void)
	{
	if (m.word_size < 2) m.word_size = 2;
	if (m.word_size > BIN_SIZE) m.word_size = BIN_SIZE;
	if (m.word_size == BIN_SIZE)
		bin_mask = ~0L;
	else	bin_mask = (1L << m.word_size) - 1;
	}


/* ------------------------------------------------------------ */

/* Clear registers.  With no argument, clear X.   With an argument of
1, clear stack.  With argument of 2, clear memory registers only. With
argument of 4, clear all. */

void
MClear(void)
	{
	int cnt;

	if (isuarg) {
		if (uarg == 1) {	/* just stack */
			for (cnt = 0; cnt < NUMSTACK; cnt++) {
				AZero(&m.r[X + cnt]);
				}
			}
		else if (uarg == 2) {	/* just memory */
			for (cnt = 0; cnt < REGCOUNT; cnt++) {
				AZero(&m.r[cnt]);
				}
			}
		else if (uarg == 4) {	/* all */
			for (cnt = 0; cnt < NUMREGS; cnt++) {
				AZero(&m.r[cnt]);
				}
			}
		}
	else	{
		MClearX();
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Clear the stack. */

void
MClearStk(void)
	{
	int cnt;

	for (cnt = 0; cnt < NUMSTACK; cnt++) {
		AZero(&m.r[X + cnt]);
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Clear X. */

void
MClearX(void)
	{
	AZero(&m.r[X]);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Set the complement mode.  With arg, set to that.  Otherwise cycle. */

void
MCompl(void)
	{
	if (isuarg) {
		m.compl_mode = uarg;
		}
	else	{
		m.compl_mode++;
		}
	if (m.compl_mode > 2) m.compl_mode = 0;
	DChangedView();
	DChanged(US_BOTTOMBAR);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Convert X to the type specified by the argument. */

void
MConv(void)
	{
	if (!isuarg) {
		DError(RES_EXPLARG);
		}
	else	{
		if (mconv_uarg == -1) {
			mconv_uarg = uarg / UNIT_SIZE;
			redo = RES_MCONV;
			}
		else	{
			UToType(mconv_uarg, &x);
			mconv_uarg = -1;
			}
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Copy X to the Copy buffer. */

void
MCopy(void)
	{
#if defined(SYSMGR)
	JCopy();
#endif
	m.r[CLIP] = m.r[X];
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Handle the Control-X prefix. */

void
MCtrlX(void)
	{
	table = TabTable(key, table);
	do	{
		KDelayPrompt(RES_PROMPTCTX);
		key = KGetChar();
		if (key == KEYQUIT) return;
		} while (key == KEYREGEN);
	TabDispatch(key, table);
	}


/* ------------------------------------------------------------ */

/* Cut X to the Copy buffer. */

void
MCut(void)
	{
	MCopy();
	MClearX();
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Do the function key specified by the arg. */

void
MDoKey(void)
	{
	if (!isuarg) {
		DError(RES_EXPLARG);
		}
	else	{
		KPush(0x100 | uarg);
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Handle the Enter key. */

void
MEnter(void)
	{
	char buf[SMALLBUFFSIZE];
	int cnt;

/* check for no command line */

	if (m.line[0] == NUL) return;

/* check for word command */

	for (cnt = Res_Number(RES_WORD_CMDS, -1) - 1 - 2; cnt >= 0; cnt -= 2) {
		if (strequ(m.line, Res_String(NULL, RES_WORD_CMDS, cnt))) {
			redo = Res_Number(RES_WORD_CMDS, cnt + 1);
			*m.line = NUL;
			DChanged(US_ENTRY);
			return;
			}
		}

/* check for unit */

	for (cnt = num_units - 1; cnt >= 0; cnt--) {
		xstrcpy(buf, units[cnt].name);
		*sindex(buf, SP) = NUL;
		if (strequ(m.line, buf)) {
			m.cur_type = cnt;
			DChanged(US_BOTTOMBAR);
			*m.line = NUL;
			DChanged(US_ENTRY);
			return;
			}
		}

/* must be number */

	if (UEnter(&x, m.line, m.cur_type)) {
		m.stack_lift = TRUE;
		}
	m.line[0] = NUL;
	DChanged(US_ENTRY);
	}


/* ------------------------------------------------------------ */

/* Enter an ASCII value. */

void
MEnterASCII(void)
	{
	char buf[LINEBUFFSIZE];
	KEYCODE key;

	do	{
		DEcho(Res_String(NULL, RES_MSGS, RES_PROMPTKEY));
		} while ((key = KGetChar()) == KEYREGEN);

	x.i = key;
	x.type = RES_UN_ASCII / UNIT_SIZE;
	m.stack_lift = TRUE;
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Exit. */

void
MExit(void)
	{
	uarg = 0;
	FEnvSave();
	FNumL(m.endnumlock);
	doabort = TRUE;
	}


/* ------------------------------------------------------------ */

/* Cycle/set the font. */

void
MFont(void)
	{
	char *cptr = Res_String(NULL, RES_CONF, RES_TYPELIST);
	char *cptr2;

	if (!isuarg) {
		for (cptr2 = cptr; *cptr2 != 101; cptr2++) {
			if (*cptr2 == screen_type) break;
			}
		if (*cptr2 == screen_type) cptr2++;
		if (*cptr2 == 101) cptr2 = cptr;
		uarg = *cptr2;
		}
	else if (uarg == 103) {
		uarg = Res_Number(RES_CONF, RES_SCRNTYPE);
		}
	else if (KAsk(RES_ASKSURE) != 'Y') {
		uarg = 0;
		return;
		}
	DFont(uarg, FALSE);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Set the hex letters mode. */

void
MHexLetters(void)
	{
	if (isuarg) {
		m.hex_letters = uarg != 0;
		}
	else	{
		m.hex_letters = !m.hex_letters;
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Self-insert. */

void
MInsChar(void)
	{
	int x = strlen(m.line);

	if (x >= sizeof(m.line) - 1) {
		TBell();
		}
	else	{
		m.line[x] = key;
		m.line[x + 1] = NUL;
		DChanged(US_ENTRY);
		}
	}


/* ------------------------------------------------------------ */

/* Evaluate a keyboard macro.  */

void
MMacEval(void)
	{
	int num;

	if (xisdigit(key)) {
		num = key - '0';
		if (num == 0) num = 10;
		num--;
		}
	else if (key >= (0x100 + 120) && key <= (0x100 + 128)) {
		num = key - (0x100 + 120) + 1;
		}
	else	{	/* default to 0 */
		num = 0;
		}
	KMacDo(RES_KEYM, num, TRUE);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Execute the menu by number. */

void
MMenu(void)
	{
	KMenu(isuarg ? uarg : 0);
	}


/* ------------------------------------------------------------ */

/* Execute the main menus. */

void
MMenuM(void)
	{
	KMenu(0);
	}


/* ------------------------------------------------------------ */

/* Handle the Meta prefix. */

void
MMeta(void)
	{
	table = TabTable(key, table);
	do	{
		KDelayPrompt(RES_PROMPTESC);
		key = KGetChar();
		if (key == KEYQUIT) return;
		} while (key == KEYREGEN);
	TabDispatch(key, table);
	}


/* ------------------------------------------------------------ */

/* Not implemented. */

void
MNotImpl(void)
	{
	DError(RES_UNKCMD);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Control Beginning NumLock setting. */

void
MNumBeg(void)
	{
	if (!isuarg) {
		m.begnumlock++;
		}
	 else	{
		m.begnumlock = uarg;
		}
	if (m.begnumlock > 2) m.begnumlock = 0;
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Control Ending NumLock setting. */

void
MNumEnd(void)
	{
	if (!isuarg) {
		m.endnumlock++;
		}
	 else	{
		m.endnumlock = uarg;
		}
	if (m.endnumlock > 3) m.endnumlock = 0;
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Advance to the next pane.  With an argument, set that as the pane. */

void
MPane(void)
	{
	if (!isuarg) {
		m.fkey_pane++;
		}
	 else	{
		m.fkey_pane = uarg / PANE_SIZE;
		}
	if (m.fkey_pane >= num_panes) m.fkey_pane = 0;
	DClear(TMaxRow() - 2, TMaxRow() - 1);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Handle a pane key press. */

void
MPaneKey(void)
	{
	int num;

	if (xisdigit(key)) {
		num = key - '0';
		if (num == 0) num = 10;
		num--;
		}
	else if (key >= (0x100 + 59) && key <= (0x100 + 68)) {
		num = key - (0x100 + 59);
		}
	else	{	/* default to F10 */
		num = 9;
		}
	KMacDo(RES_PANES, m.fkey_pane * PANE_SIZE + num * 2 + 1, FALSE);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Paste from the Copy buffer. */

void
MPaste(void)
	{
#if defined(SYSMGR)
	struct number oldx;

	oldx = m.r[X];
	if (JPaste()) {
		if (m.stack_lift) {
			m.r[T] = m.r[Z];
			m.r[Z] = m.r[Y];
			m.r[Y] = oldx;
			}
		m.stack_lift = TRUE;
		}
#else
	if (m.stack_lift) {
		m.r[T] = m.r[Z];
		m.r[Z] = m.r[Y];
		m.r[Y] = m.r[X];
		}
	m.r[X] = m.r[CLIP];
	m.stack_lift = TRUE;
#endif
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Quote the next character. */

void
MQuote(void)
	{
	KEYCODE c;

	do	{
		KDelayPrompt(RES_PROMPTQUOTE);
		c = KGetChar();
		if (c == KEYQUIT) return;
		} while (c == KEYREGEN);
	while (uarg-- > 0) {
		key = c;
		MInsChar();
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Control Radix mark setting. */

void
MRadix(void)
	{
	if (!isuarg) {
		m.radix++;
		}
	 else	{
		m.radix = uarg;
		}
	if (m.radix > 1) m.radix = 0;
	DChangedView();
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Show the registers. */

void
MRegisters(void)
	{
	char buf[COLMAX * 2];
	char smallbuf[SMALLBUFFSIZE];
	int cnt;

	do	{
		buf[0] = NUL;
		DLine2(buf, 0, TAttrNorm());

		for (cnt = 1; cnt < REGCOUNT + 1; cnt++) {
			xsprintf(smallbuf,
				Res_String(NULL, RES_MSGS, RES_REGSTR),
				cnt - 1);
			DFormat(buf, &m.r[cnt - 1], smallbuf);
			DLine2(buf, cnt, TAttrNorm());
			}
		TSetPoint(cnt, 0);
		DRowChar(RES_OTHERSPLITCHAR);
		TSetPoint(cnt + 1, 0);
		TForce();
		DClear(0, cnt);
		} while (KGetChar() == KEYREGEN);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Set the tab spacing to the argument. */

void
MSetTabs(void)
	{
	if (!isuarg) {
		DError(RES_EXPLARG);
		}
	else	{
		m.tab_spacing = uarg;
		DChangedView();
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Set the show status of the specified type. */

void
MShow(void)
	{
	if (!isuarg) {
		DError(RES_EXPLARG);
		}
	else	{
		if (uarg > (num_units * UNIT_SIZE)) uarg = 0;
		m.show_types[uarg / UNIT_SIZE] =
			!m.show_types[uarg / UNIT_SIZE];
		DChangedView();
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Handle the space key. */

void
MSpace(void)
	{
	MEnter();
	}


/* ------------------------------------------------------------ */

/* Show the status. */

void
MStatus(void)
	{
	char buf[COLMAX * 2];
	int cnt;
	int cnt2;

	do	{
		xsprintf(buf, Res_String(NULL, RES_MSGS, RES_STATUSSTR1),
			REGCOUNT, m.radix);
		DLine2(buf, 0, TAttrNorm());

		xsprintf(buf, Res_String(NULL, RES_MSGS, RES_STATUSSTR2),
			m.stack_lift, m.hex_letters);
		DLine2(buf, 1, TAttrNorm());

		xsprintf(buf, Res_String(NULL, RES_MSGS, RES_STATUSSTR3),
			m.begnumlock, m.endnumlock);
		DLine2(buf, 2, TAttrNorm());

		xsprintf(buf, Res_String(NULL, RES_MSGS, RES_STATUSSTR4),
			m.tab_spacing);
		DLine2(buf, 3, TAttrNorm());

		xstrcpy(buf, Res_String(NULL, RES_MSGS, RES_STATUSSTR5));
		DLine2(buf, 4, TAttrNorm());
		*buf = NUL;
		DLine2(buf, 5, TAttrNorm());

		DLine2(buf, 6, TAttrNorm());
		DLine2(buf, 7, TAttrNorm());
		DLine2(buf, 8, TAttrNorm());
		DLine2(buf, 9, TAttrNorm());

		*buf = NUL;
		cnt2 = 6;
		for (cnt = num_units - 1; cnt >= 0; cnt--) {
			if (m.show_types[cnt]) {
				strcat(buf, units[cnt].name);
				strcat(buf, " ");

				if (strlen(buf) > TMaxCol() - 8) {
					DLine2(buf, cnt2++, TAttrNorm());
					*buf = NUL;
					}
				}
			}

		if (*buf != NUL) DLine2(buf, cnt2, TAttrNorm());

		*buf = NUL;
		DLine2(buf, 10, TAttrNorm());
		xstrcpy(buf, Res_String(NULL, RES_MSGS, RES_STATUSSTR6));
		DLine2(buf, 11, TAttrNorm());
		DFormat(buf, &m.r[CLIP], "C");
		DLine2(buf, 12,  TAttrNorm());

		TSetPoint(13, 0);
		DRowChar(RES_OTHERSPLITCHAR);
		TSetPoint(14, 0);
		TForce();
		DClear(0, 14);
		} while (KGetChar() == KEYREGEN);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Convert the value in STR to a decimal number and return it in N. 
The number is in base BASE.  Return TRUE if the conversion was
successful, FALSE if the string was not a valid number.

This routine allows for leading whitespace and can handle leading + or
- signs. */

FLAG
MSToNL(char *str, long *n, int base)
	{
	unsigned long val;
	int minus;
	char chr;

	while (*str == SP || *str == TAB) str++;
	if (*str == '-') {
		minus = -1;
		str++;
		}
	else	{
		minus = 1;
		if (*str == '+') ++str;
		}
	for (val = 0; *str; ++str) {
		chr = xtoupper(*str);
		if (xisalpha(chr)) chr -= 'A' - 10;
		else if (xisdigit(chr)) chr -= '0';
		else return(FALSE);
		if (chr >= base) return(FALSE);
		val = val * base + chr;
		}
	*n = val * minus;
	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Set the current type. */

void
MType(void)
	{
	if (!isuarg) {
		DError(RES_EXPLARG);
		}
	else	{
		m.cur_type = uarg / UNIT_SIZE;
		DChanged(US_BOTTOMBAR);
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Handle argument prefix. */

void
MUArg(void)
	{
	FLAG numflag = FALSE;

	uarg *= 4;
	do	{
		KUArg(uarg);
		while (xisdigit(key = KGetChar()))  {
			if (!numflag) {
				uarg = 0;
				numflag = TRUE;
				}
			uarg = uarg * 10 + key - '0';
			KUArg(uarg);
			}
		} while (key == KEYREGEN);
	isuarg = TRUE;
	TabDispatch(key, table);
	}


/* ------------------------------------------------------------ */

/* Set the word size. */

void
MWSize(void)
	{
	if (!isuarg) {
		DError(RES_EXPLARG);
		}
	else	{
		m.word_size = uarg;
		MBinFix();
		DChangedView();
		DChanged(US_BOTTOMBAR);
		}
	uarg = 0;
	}


/* end of LMISC.C -- Miscellaneous Commands */
