/* LDISPLAY.C -- Redisplay and Display Routines

	Written April 1994 by Craig A. Finseth
	Copyright 1994 by Craig A. Finseth
*/

#include "loki.h"

static enum USES line_usage[ROWMAX];	/* how each line is used */
static int line_use[(int) US_PANE2];
static FLAG line_bad[ROWMAX];		/* lines to be re-shown */

static FLAG needclear = TRUE;

static struct number prev_x;
static struct number prev_y;
static struct number prev_z;
static struct number prev_t;
static struct number prev_l;
static struct number prev_m;

#define X_ROW	(TMaxRow() - 5)
#define Y_ROW	(TMaxRow() - 6)
#define Z_ROW	(TMaxRow() - 7)
#define T_ROW	(TMaxRow() - 8)
#define L_ROW	(TMaxRow() - 10)
#define M_ROW	(TMaxRow() - 12)

static char mode_buf[COLMAX + 1];

void D_ClearScreen(void);

/* ------------------------------------------------------------ */

/* Initialize the display routines, part 1. */

void
DInit(void)
	{
	int cnt;

	D_ClearScreen();
	for (cnt = 0; cnt < ROWMAX; cnt++) {
		line_usage[cnt] = US_BLANK;
		}
	line_usage[0] = US_HEAD;
	line_usage[1] = US_TOPBAR;
	line_usage[M_ROW] = US_M;
	line_usage[L_ROW] = US_L;
	line_usage[T_ROW] = US_T;
	line_usage[Z_ROW] = US_Z;
	line_usage[Y_ROW] = US_Y;
	line_usage[X_ROW] = US_X;
	line_usage[TMaxRow() - 4] = US_BOTTOMBAR;
	line_usage[TMaxRow() - 3] = US_ENTRY;
	line_usage[TMaxRow() - 2] = US_PANE1;
	line_usage[TMaxRow() - 1] = US_PANE2;

	for (cnt = 0; cnt < sizeof(line_use) / sizeof(line_use[0]); cnt++) {
		line_use[cnt] = 0;
		}
	line_use[US_M] = M_ROW;
	line_use[US_L] = L_ROW;
	line_use[US_T] = T_ROW;
	line_use[US_Z] = Z_ROW;
	line_use[US_Y] = Y_ROW;
	line_use[US_X] = X_ROW;
	line_use[US_BOTTOMBAR] = TMaxRow() - 4;
	line_use[US_ENTRY] = TMaxRow() - 3;
	}


/* ------------------------------------------------------------ */

/* Terminate the display routines. */

void
DFini(void)
	{
	TSetPoint(TMaxRow() - 1, 0);
	THiOff();
	TCLEOL();
#if defined(MSDOS)
#if !defined(SYSMGR)
	if (screen_type == 16) TSetPoint(24, 0);
#endif
#endif
	}


/* ------------------------------------------------------------ */

/* Mark the specified row as changed. */

void
DChanged(enum USES u)
	{
	line_bad[line_use[(int) u]] = TRUE;
	}


/* ------------------------------------------------------------ */

/* Mark the displayed numbers as changed. */

void
DChangedView(void)
	{
	DChanged(US_X);
	DChanged(US_Y);
	DChanged(US_Z);
	DChanged(US_T);
	DChanged(US_L);
	DChanged(US_M);
	}


/* ------------------------------------------------------------ */

/* Clear the row. */

void
DClear(int start, int stop)
	{
	while (start <= stop) {
		line_bad[start++] = TRUE;
		}
	}


/* ------------------------------------------------------------ */

/* Display a message in the echo line. */

void
DEcho(char *msg)
	{
	TSetPoint(0, 0);
	THiOn();

	strncpy(mode_buf, msg, TMaxCol() + 1);
	mode_buf[TMaxCol() - 1] = NUL;
	TPutStr(mode_buf);

	TCLEOL();
	THiOff();
	TForce();
	DClear(0, 0);
	}


/* ------------------------------------------------------------ */

/* Display a message in the echo line, but don't move the cursor. */

void
DEchoNM(char *msg)
	{
	int row;
	int col;

	row = TGetRow();
	col = TGetCol();
	DEcho(msg);
	TSetPoint(row, col);
	TForce();
	}


/* ------------------------------------------------------------ */

/* Display an error message */

void
DError(int msgnum)
	{
	DErrorStr(Res_String(NULL, RES_MSGS, msgnum));
	}


/* ------------------------------------------------------------ */

/* Display an error message */

void
DErrorStr(char *msg)
	{
	char buf[2 * COLMAX + 1];
	int row;
	int col;

	row = TGetRow();
	col = TGetCol();

	xsprintf(buf, ">>> %s", msg);
	TBell();

	do	{
		DEcho(buf);
		} while (KGetChar() == KEYREGEN);

	TSetPoint(row, col);
	TForce();
	}


/* ------------------------------------------------------------ */

/* Set the font to NEW.  If -1, turn off any graphics.  Return the old
font. Do an IncrDisplay if DOINCR is True. */

int
DFont(int new, FLAG doincr)
	{
	int old = screen_type;

	DFini();
	TFini();
	screen_type = new;
	TInit();
	DInit();
	needclear = FALSE;
	DNewDisplay();
	if (doincr) DIncrDisplay();
	return(old);
	}


/* ------------------------------------------------------------ */

/* Format a number line. */

void
DFormat(char *buf, struct number *nptr, char *which)
	{
	char ubuf[UNIT_WIDTH];
	char bigbuf[BUFFSIZE];
	char *cptr = bigbuf;
	int cnt;
	int amt;
	int len;
	int cat = units[nptr->type].cat;

	for (cnt = num_units - 1; cnt >= 0; cnt--) {
		if (nptr->type == cnt ||
			 (units[cnt].cat == cat &&
			 m.show_types[cnt])) {
			UFormat(ubuf, cnt, nptr);

			len = strlen(ubuf);
			amt = TGetTabWidth(len + 1);
			memset(cptr, SP, amt);
			cptr += amt;
			memmove(cptr, ubuf, len);
			cptr += len;
			*cptr++ = (nptr->type == cnt) ? '!' : SP;
			}
		}

	*cptr = NUL;

	memset(buf, SP, TMaxCol());
	buf[TMaxCol()] = NUL;
	memmove(buf + TMaxCol() - 3, which, strlen(which));

	len = strlen(bigbuf);
	cnt = min(len, TMaxCol() - 3);
	memmove(buf + TMaxCol() - 3 - cnt, bigbuf + len - cnt, cnt);
	}

			
/* ------------------------------------------------------------ */

/* Update the display from the buffer */

void
DIncrDisplay(void)
	{
	int cnt;

#if defined(MSDOS)
	if (madefor == 'C' || madefor == 'D' || madefor == 'J' ||
		madefor == 'S') JDisStart();
#endif

	if (!line_bad[X_ROW] && !ASame(&m.r[X], &prev_x)) {
		line_bad[X_ROW] = TRUE;
		prev_x = m.r[X];
		}
	if (!line_bad[Y_ROW] && !ASame(&m.r[Y], &prev_y)) {
		line_bad[Y_ROW] = TRUE;
		prev_y = m.r[Y];
		}
	if (!line_bad[Z_ROW] && !ASame(&m.r[Z], &prev_z)) {
		line_bad[Z_ROW] = TRUE;
		prev_z = m.r[Z];
		}
	if (!line_bad[T_ROW] && !ASame(&m.r[T], &prev_t)) {
		line_bad[T_ROW] = TRUE;
		prev_t = m.r[T];
		}
	if (!line_bad[L_ROW] && !ASame(&m.r[L], &prev_l)) {
		line_bad[L_ROW] = TRUE;
		prev_l = m.r[L];
		}
	if (!line_bad[M_ROW] && !ASame(&m.r[0], &prev_m)) {
		line_bad[M_ROW] = TRUE;
		prev_m = m.r[0];
		}

	for (cnt = TMaxRow() - 1; cnt >= 0; cnt--) {
		if (line_bad[cnt]) {
			DLine(cnt);
			line_bad[cnt] = FALSE;
			if (KIsKey() == 'Y') break;
			}
		}

#if defined(MSDOS)
	if (madefor == 'C' || madefor == 'D' || madefor == 'J' ||
		madefor == 'S') JDisEnd();
#endif
	TSetPoint(TMaxRow() - 3, TMaxCol() - 3);
	}


/* ------------------------------------------------------------ */

/* Force a redisplay of the specified line.  The line is TMaxCol()
chars wide. */

void
DLine(int row)
	{
	char buf[2 * COLMAX + 1];
	char buf2[2 * COLMAX + 1];
	int cnt;
	int amt;
	int len;
	char *cptr;

	switch (line_usage[row]) {
	
	case US_HEAD:
		xstrcpy(buf, Res_String(NULL, RES_MSGS, RES_HEADER));
		DLine2(buf, row, TAttrNorm());
		break;

	case US_TOPBAR:
		DRowChar2(row, RES_MENU0SPLITCHAR, TAttrNorm());
		break;

	case US_BLANK:
		*buf = NUL;
		DLine2(buf, row, TAttrNorm());
		break;

	case US_M:
		DFormat(buf, &m.r[0], "M");
		DLine2(buf, row, TAttrNorm());
		break;

	case US_L:
		DFormat(buf, &m.r[L], "L");
		DLine2(buf, row, TAttrNorm());
		break;

	case US_T:
		DFormat(buf, &m.r[T], "T");
		DLine2(buf, row, TAttrNorm());
		break;

	case US_Z:
		DFormat(buf, &m.r[Z], "Z");
		DLine2(buf, row, TAttrNorm());
		break;

	case US_Y:
		DFormat(buf, &m.r[Y], "Y");
		DLine2(buf, row, TAttrNorm());
		break;

	case US_X:
		DFormat(buf, &m.r[X], "X");
		DLine2(buf, row, TAttrNorm());
		break;

	case US_BOTTOMBAR:
		memset(buf, Res_Number(RES_CONF, RES_WINDSPLITCHAR),
			TMaxCol());

		xsprintf(buf2, " %s ", units[m.cur_type].name);
		amt = strlen(buf2);
		cptr = buf + 3;
		memmove(cptr, buf2, amt);
		cptr += amt + 3;

		xsprintf(buf2, Res_String(NULL, RES_MSGS, RES_DISPBOT1),
			m.compl_mode, m.word_size);
		amt = strlen(buf2);
		memmove(cptr, buf2, amt);
		cptr += amt + 3;

		xsprintf(buf2, Res_String(NULL, RES_MSGS, RES_DISPBOT2),
			m.frac_mode, m.frac_denom);
		amt = strlen(buf2);
		memmove(cptr, buf2, amt);

		DLine2(buf, row, TAttrNorm());
		break;

	case US_ENTRY:
		memset(buf, SP, TMaxCol());
		if (m.line[0] != NUL) {
			memmove(buf + TMaxCol() - strlen(m.line) - 3, m.line,
				strlen(m.line));
			}
		DLine2(buf, row, TAttrNorm());
		break;

	case US_PANE1:
		memset(buf, SP, TMaxCol());
		amt = TMaxCol() / NUM_FUNCS;
		for (cnt = 0; cnt < NUM_FUNCS; cnt += 2) {
			cptr = Res_String(&len, RES_PANES,
				m.fkey_pane * PANE_SIZE + cnt * PANE_KEY_SIZE);
			memmove(&buf[TMaxCol() * cnt / NUM_FUNCS], cptr,
				min(len - 1, 2 * (amt - 1)));
			}
		DLine2(buf, row, TAttrRev());
		break;

	case US_PANE2:
		memset(buf, SP, TMaxCol());
		amt = TMaxCol() / NUM_FUNCS;
		for (cnt = 1; cnt < NUM_FUNCS; cnt += 2) {
			cptr = Res_String(&len, RES_PANES,
				m.fkey_pane * PANE_SIZE + cnt * PANE_KEY_SIZE);
			memmove(&buf[TMaxCol() * cnt / NUM_FUNCS], cptr,
				min(len - 1, 2 * (amt - 1)));
			}
		DLine2(buf, row, TAttrRev());
		break;
		}
	}


/* ------------------------------------------------------------ */

/* Display a line.  CPTR must point to a MODIFIABLE buffer at leaset
TMaxCol() + 1 characters long. */

void
DLine2(char *cptr, int row, int attr)
	{
	TSetPoint(row, 0);
	if (attr != TAttrNorm()) THiOn();
	*(cptr + TMaxCol()) = NUL;
	TPutStr(cptr);
	TCLEOL();
	if (attr != TAttrNorm()) THiOff();
	TSetPoint(row, 0);
	}


/* ------------------------------------------------------------ */

/* Put the cursor at the WHICHth entry. */

void
DMenu(int which, int *rows, int *cols)
	{
	TSetPoint(rows[which], cols[which]);
	}


/* ------------------------------------------------------------ */

/* Set up the menu.  Display the menu numbered MENU, starting at entry
FIRST for NUMBER entries.  Put the row, column, and hot key
information in ROWS, COLS, and HOT_CHARS. */

void
DMenuSetup(int *rows, int *cols, char *hot_chars, int menu, int first,
	int number)
	{
	int cnt;
	int chr;
	int row;
	int wid;
	int maxwid;
	int llen;
	int cmdlen;
	char *cmdptr;
	FLAG isand;
	FLAG nexthot;
	char *lptr;
	char *boxchars = Res_String(NULL, RES_CONF, RES_BOXCHARS);
	char dispbuf[(COLMAX + 1) * 2];
	char label[SMALLBUFFSIZE];
	char *cptr;
	unsigned char *rptr;
	int type = Res_Number(menu, 1);

	if (!(type & 0x10)) {	/* not a floating menu */
		float_row = 0;
		float_col = 0;
		}
	else	{
		TSetPoint(float_row, float_col);
		}
	type &= 0x0f;

	switch (type) {

	case 0:
	case 1:
		row = float_row;
		if (type == 1) {
			cptr = dispbuf;
			*cptr++ = boxchars[0];
			*cptr++ = TAttrNorm();
			for (cnt = 1; cnt < TMaxCol() - 1; cnt++) {
				*cptr++ = boxchars[1];
				*cptr++ = TAttrNorm();
				}
			*cptr++ = boxchars[2];
			*cptr   = TAttrNorm();
			TSetPoint(row++, 0);
			TAttrBlock(dispbuf, TMaxCol(), FALSE, 0);
			}		
		for (cnt = first; cnt < number; row++) {
			cptr = dispbuf;
			if (type == 1) {
				*cptr++ = boxchars[3];
				*cptr++ = TAttrNorm();
				}
			for (; cnt < number; cnt++) {
				*cptr++ = SP;
				*cptr++ = TAttrNorm();

				xstrncpy(label, Res_String(NULL, menu,
					cnt * 2 + 2));
				isand = FALSE;
				for (lptr = label, llen = 0; *lptr != NUL;
					 lptr++) {
					if (*lptr == '\\') {
						llen++;
						lptr++;
						if (*lptr == NUL) lptr--;
						}
					else if (*lptr == '&') {
						isand = TRUE;
						}
					else	{
						llen++;
						}
					}
				if ((cptr - dispbuf) / 2 + llen >
					 TMaxCol() - ((type == 1) ? 3 : 2))
					break;

				*rows++ = row;
				*cols++ = (cptr - dispbuf) / 2;
				nexthot = !isand;
				for (lptr = label, llen = 0; *lptr != NUL;
					 lptr++) {
					if (nexthot) {
						if (*lptr == '\\') {
							lptr++;
							if (*lptr == NUL)
								lptr--;
							}
						*cptr++ = *lptr;
						*hot_chars++ =
							ConvUpper(*lptr);
						*cptr++ = TAttrUnder();
						nexthot = FALSE;
						}
					else if (*lptr == '\\') {
						lptr++;
						if (*lptr == NUL) lptr--;
						*cptr++ = *lptr;
						*cptr++ = TAttrNorm();
						}
					else if (*lptr == '&') {
						nexthot = TRUE;
						}
					else	{
						*cptr++ = *lptr;
						*cptr++ = TAttrNorm();
						}
					}

				*cptr++ = SP;
				*cptr++ = TAttrNorm();
				}
			while ((cptr - dispbuf) / 2 < TMaxCol() - 1) {
				*cptr++ = SP;
				*cptr++ = TAttrNorm();
				}
			*cptr++ = (type == 1) ? boxchars[4] : SP;
			*cptr++ = TAttrNorm();
			TSetPoint(row, 0);
			TAttrBlock(dispbuf, TMaxCol(), FALSE, 0);
			}
		TSetPoint(row, 0);

		if (type == 0) {
			chr = *Res_String(NULL, RES_CONF, RES_MENU0SPLITCHAR);
			for (cptr = dispbuf; cptr < &dispbuf[5]; cptr++)
				*cptr = chr;
			xstrcpy(cptr, Res_String(NULL, menu, 0));
			cptr += strlen(cptr);
			while (cptr - dispbuf < TMaxCol()) *cptr++ = chr;
			*cptr = NUL;
			TPutStr(dispbuf);
			}
		else	{
			cptr = dispbuf;
			*cptr++ = boxchars[5];
			*cptr++ = TAttrNorm();
			for (cnt = 1; cnt < TMaxCol() - 1; cnt++) {
				*cptr++ = boxchars[6];
				*cptr++ = TAttrNorm();
				}
			*cptr++ = boxchars[7];
			*cptr   = TAttrNorm();
			TSetPoint(row++, 0);
			TAttrBlock(dispbuf, TMaxCol(), FALSE, 0);
			}		
		break;

	case 2:
		TSetPoint(float_row, float_col);
		THiOn();
		xstrncpy(label, Res_String(NULL, menu, 0));
		for (lptr = label; *lptr == SP; lptr++) ;
		for (cptr = lptr; *cptr != SP; cptr++) ;
		*cptr = NUL;
		TPutStr(lptr);
		THiOff();

		/* find width */

		maxwid = 0;
		for (cnt = first; cnt < number; cnt++) {
			wid = 4;
			for (cptr = Res_String(NULL, menu, cnt * 2 + 2);
				 *cptr != NUL; cptr++) {
				if (*cptr == '|')
					wid += 3;
				else if (*cptr == '\\') {
					cptr++;
					if (*cptr == NUL) cptr--;
					wid++;
					}
				else if (*cptr != '&')
					wid++;
				}
			if (wid > maxwid) maxwid = wid;
			}

		if (float_col > TMaxCol() - maxwid) {
			float_col = TMaxCol() - maxwid;
			}

		row = float_row + 1;

		cptr = dispbuf;
		*cptr++ = boxchars[0];
		*cptr++ = TAttrNorm();
		for (cnt = 2; cnt < maxwid; cnt++) {
			*cptr++ = boxchars[1];
			*cptr++ = TAttrNorm();
			}
		*cptr++ = boxchars[2];
		*cptr   = TAttrNorm();
		TSetPoint(row++, float_col);
		TAttrBlock(dispbuf, maxwid, FALSE, 0);

		for (cnt = first; cnt < number; row++, cnt++) {
			cptr = dispbuf;
			*cptr++ = boxchars[3];
			*cptr++ = TAttrNorm();
			*cptr++ = SP;
			*cptr++ = TAttrNorm();

			xstrncpy(label, Res_String(NULL, menu, cnt * 2 + 2));
			isand = FALSE;
			cmdptr = NULL;
			for (lptr = label; *lptr != NUL; lptr++) {
				if (*lptr == '\\') {
					lptr++;
					if (*lptr == NUL) lptr--;
					}
				else if (*lptr == '&')
					isand = TRUE;
				else if (*lptr == '|')
					cmdptr = lptr;
				}
			cmdlen = (cmdptr != NULL) ? lptr - cmdptr : 0;

			*rows++ = row;
			*cols++ = float_col + 2;

			nexthot = !isand;
			for (lptr = label, llen = 0; *lptr != NUL; lptr++) {
				if (nexthot) {
					if (*lptr == '\\') {
						lptr++;
						if (*lptr == NUL) lptr--;
						}
					*cptr++ = *lptr;
					*hot_chars++ = ConvUpper(*lptr);
					*cptr++ = TAttrUnder();
					nexthot = FALSE;
					llen++;
					}
				else if (*lptr == '\\') {
					lptr++;
					if (*lptr == NUL) lptr--;
					*cptr++ = *lptr;
					*cptr++ = TAttrNorm();
					llen++;
					}
				else if (*lptr == '&') {
					nexthot = TRUE;
					}
				else if (*lptr == '|') {
					while (llen < maxwid - 3 - cmdlen) {
						*cptr++ = SP;
						*cptr++ = TAttrNorm();
						llen++;
						}
					}
				else	{
					*cptr++ = *lptr;
					*cptr++ = TAttrNorm();
					llen++;
					}
				}

			while (llen++ < maxwid - 3) {
				*cptr++ = SP;
				*cptr++ = TAttrNorm();
				}
			*cptr++ = boxchars[4];
			*cptr++ = TAttrNorm();
			TSetPoint(row, float_col);
			TAttrBlock(dispbuf, maxwid, FALSE, 0);
			}

		cptr = dispbuf;
		*cptr++ = boxchars[5];
		*cptr++ = TAttrNorm();
		for (cnt = 2; cnt < maxwid; cnt++) {
			*cptr++ = boxchars[6];
			*cptr++ = TAttrNorm();
			}
		*cptr++ = boxchars[7];
		*cptr   = TAttrNorm();
		TSetPoint(row++, float_col);
		TAttrBlock(dispbuf, maxwid, FALSE, 0);
		break;
		}
	TSetPoint(TGetRow() + 1, 0);
	}


/* ------------------------------------------------------------  */

/* Put a new display on the screen. */

void
DNewDisplay(void)
	{
	if (needclear) TClrScreen();
	needclear = TRUE;
	D_ClearScreen();
	}


/* ------------------------------------------------------------ */

/* Display the date/time in the upper right. */

void
DPostTime(void)
	{
	char buf[BUFFSIZE];
	struct tm t;
	int row;
	int col;

	row = TGetRow();
	col = TGetCol();

	DNow(&t);
	xsprintf(buf, "%4d-%02d-%02d %02d:%02d:%02d",
		t.tm_year, t.tm_mon + 1, t.tm_mday,
		t.tm_hour, t.tm_min, t.tm_sec);

	TSetPoint(0, TMaxCol() - strlen(buf) - 1);
	TPutStr(buf);

	TSetPoint(row, col);
	TForce();
	}


/* ------------------------------------------------------------ */

/* Display a row of the specified character. */

void
DRowChar(int which)
	{
	int chr = *Res_String(NULL, RES_CONF, which);
	int cnt;

	THiOn();
	for (cnt = 0; cnt < TMaxCol(); ++cnt) TPutChar(chr);
	THiOff();
	}


/* ------------------------------------------------------------ */

/* Display a row of the specified character. */

void
DRowChar2(int row, int which, int attr)
	{
	char buf[COLMAX + 1];
	int cnt;
	int chr = *Res_String(NULL, RES_CONF, which);

	for (cnt = 0; cnt < TMaxCol(); ++cnt) {
		buf[cnt] = chr;
		}
	buf[cnt] = NUL;
	DLine2(buf, row, attr);
	}


/* ------------------------------------------------------------ */

/* Show the multi-line message starting at WHERE (T=top of screen,
M=after menu bar).  Turn on highlighting if HILIT is True. */

void
DShow(char where, char *str, FLAG hilit)
	{
	unsigned char buf[BUFFSIZE];
	unsigned char *cptr;
	unsigned char *mptr;
	int start_row = where == 'M' ? float_row : 0;
	int row;

	row = start_row;

	if (hilit) THiOn();
	for (mptr = str; *mptr != NUL;
		 mptr = (*cptr == NL) ? (cptr + 1) : cptr) {
		cptr = (unsigned char *)sindex(mptr, NL);
		memmove(buf, mptr, cptr - mptr);
		buf[cptr - mptr] = NUL;

		TSetPoint(row++, 0);
		TPutStr(buf);
		TCLEOL();
		if (row >= TMaxRow() - 1) break;
		}
	if (!hilit && row < TMaxRow() - 1) {
		TSetPoint(row++, 0);
		DRowChar(RES_OTHERSPLITCHAR);
		}
	TSetPoint(row, 0);
	TForce();
	if (hilit) THiOff();
	DClear(start_row, TGetRow());
	}


/* ------------------------------------------------------------ */

/* Display an informative message in the message area and wait for a
key press. */

void
DView(char *msg)
	{
	int row;
	int col;

	row = TGetRow();
	col = TGetCol();
	do	{
		DEcho(msg);
		} while (KGetChar() == KEYREGEN);
	TSetPoint(row, col);
	TForce();
	}


/* ------------------------------------------------------------ */

/* Clear the screen. */

void
D_ClearScreen(void)
	{
	DClear(0, TMaxRow() - 1);
	}


/* end of LDISPLAY.C -- Redisplay and Display Routines */
