/* INI.C -- Routines to Load the Initialization File

	Written March 1991 by Craig A. Finseth
	Copyright 1991 by Craig A. Finseth
*/

#include "freyja.h"

#define PAGEK		(PAGESIZE / 1024)

/* parsing and initialization tables */

#define NUM_IC		(sizeof(ic) / sizeof(ic[0]))
static struct ini_char {
	char *tok;	/* token to match */
	char *var;	/* variable to store into */
	char def;	/* default value */
	char *valid;	/* valid characters */
	} ic[] = {
#if defined(UNIX)
{ "SCR", &c.g.screen_type,	'V',	"@VT" },	/* must be first */
{ "SCS", &c.g.screen_size,	'@',	"@" },
{ "KEY", &c.g.key_type,		'C',	"@C" },
{ "SPE", &c.g.special,		'@',	"@" },
{ "PA1", &c.g.parameter_1,	'@',	"@" },
#else
#if defined(MSDOS)
{ "SCR", &c.g.screen_type,	'V',	"@VBM" },	/* must be first */
{ "SCS", &c.g.screen_size,	'@',	"@J" },
{ "KEY", &c.g.key_type,		'C',	"@CSBJ" },
{ "SPE", &c.g.special,		'@',	"@J" },
{ "PA1", &c.g.parameter_1,	'@',	"@P" },
#else
{ "SCR", &c.g.screen_type,	'V',	"@V" },		/* must be first */
{ "SCS", &c.g.screen_size,	'@',	"@" },
{ "KEY", &c.g.key_type,		'C',	"@C" },
{ "SPE", &c.g.special,		'@',	"@" },
{ "PA1", &c.g.parameter_1,	'@',	"@" },
#endif
#endif
{ "MET", &c.g.meta_handle,	'B',	"@BIM" },
{ "FIL", &c.d.fill,		'N',	"@NFW" } };

#define NUM_IF		(sizeof(ifl) / sizeof(ifl[0]))
static struct ini_flag {
	char *tok;	/* token to match */
	FLAG *var;	/* variable to store into */
	FLAG def;	/* default value */
	} ifl[] = {
{ "WRA", &c.g.wrap_allowed,	TRUE },
{ "VIS", &c.g.vis_gray,		FALSE },
{ "USE", &c.g.use_caret,	TRUE } };

#define NUM_II		(sizeof(ii) / sizeof(ii[0]))
static struct ini_int {
	char *tok;	/* token to match */
	int *var;	/* variable to store into */
	int def;	/* default value */
	} ii[] = {
{ "SWA", &c.g.swap_size,	128 },
{ "ESC", &c.g.ESC_swap,		27 },
{ "LEF", &c.d.left_margin,	0 },
{ "RIG", &c.d.right_margin,	70 },
{ "TAB", &c.d.tab_spacing,	8 } };


static char *buffer;
static char *bufptr;
static int buflen;
static int line;
static int fd;

int Ini_Next();		/* int fd */
FLAG Ini_Token();	/* char *token, int len */

/* ------------------------------------------------------------ */

/* Load the configuration information.  If no file, use defaults.  If
an error, print a message and use defaults. */

FLAG
IniLoad(s, z)
	char s;
	int z;
	{
	char fname[FNAMEMAX];
	char buf[BUFFSIZE];
	char tok1[32];
	char tok2[32];
	char chr;
	int cnt;
	FLAG handled;

/* set defaults */

	for (cnt = 0; cnt < NUM_IC; cnt++) {
		*ic[cnt].var = ic[cnt].def;
		}
	for (cnt = 0; cnt < NUM_IF; cnt++) {
		*ifl[cnt].var = ifl[cnt].def;
		}
	for (cnt = 0; cnt < NUM_II; cnt++) {
		*ii[cnt].var = ii[cnt].def;
		}

	s = xtoupper(s);
	if (*sindex(ic[0].valid, s) == NUL) {
#if defined(SYSMGR)
		xsprintf(buf, "option value '%c' is not one of %s.",
			s, ic[0].valid);
		JMsg(buf);
#else
		xeprintf("option value '%c' is not one of %s.\r\n",
			s, ic[0].valid);
#endif
		return(FALSE);
		}
	c.g.screen_type = s;
	c.g.swap_size = z;

/* process the file */

	*fname = NUL;
	if ((fd = FPathOpen(INI_FILENAME, fname)) < 0) goto OK;

	buffer = buf;
	bufptr = buffer;
	buflen = -1;
	line = 1;

	while (Ini_Token(tok1, sizeof(tok1)) && Ini_Token(tok2, sizeof(tok2))){
		handled = FALSE;

		for (cnt = 0; cnt < NUM_IC; cnt++) {
			if (!strequ(tok1, ic[cnt].tok)) continue;
			*ic[cnt].var = xtoupper(*tok2);
			if (*sindex(ic[cnt].valid, *ic[cnt].var) == NUL) {
#if defined(SYSMGR)
				xsprintf(buf, "%s value '%c' is not one of %s.",
					ic[cnt].tok, *tok2, ic[cnt].valid);
				JMsg(buf);
#else
				xeprintf("%s value '%c' is not one of %s.\r\n",
					ic[cnt].tok, *tok2, ic[cnt].valid);
#endif
				}
			else	handled = TRUE;
			}

		for (cnt = 0; cnt < NUM_IF; cnt++) {
			if (!strequ(tok1, ifl[cnt].tok)) continue;
			chr = xtoupper(*tok2);
			if (*sindex("YN", chr) == NUL) {
#if defined(SYSMGR)
				xsprintf(buf, "%s value '%s' is not one of Y or N.",
					ifl[cnt].tok, tok2);
				JMsg(buf);
#else
				xeprintf("%s value '%s' is not one of Y or N.\r\n",
					ifl[cnt].tok, tok2);
#endif
				}
			else	{
				*ifl[cnt].var = chr == 'Y';
				handled = TRUE;
				}
			}

		for (cnt = 0; cnt < NUM_II; cnt++) {
			if (!strequ(tok1, ii[cnt].tok)) continue;
			if (!SToN(tok2, ii[cnt].var, 10)) {
#if defined(SYSMGR)
				xsprintf(buf, "%s value '%s' is not a number.",
					ii[cnt].tok, tok2);
				JMsg(buf);
#else
				xeprintf("%s value '%s' is not a number.\r\n",
					ii[cnt].tok, tok2);
#endif
				}
			else	{
				if (*ii[cnt].var < 0) *ii[cnt].var = 0;
				handled = TRUE;
				}
			}

		if (!handled) {
#if defined(SYSMGR)
			xsprintf(buf, "Problem on line %d of ini file %s.\r\n",
				line, fname);
			JMsg(buf);
#else
			xeprintf("Problem on line %d of ini file %s.\r\n",
				line, fname);
#endif
			close(fd);
			return(FALSE);
			}
		}
	close(fd);

/* handle consistency */

OK:

	for (cnt = 0; cnt < NUM_IC; cnt++) {
		if (*ic[cnt].var == '@') *ic[cnt].var = ic[cnt].def;
		}

	if (c.g.swap_size < 16) c.g.swap_size = 16;
	if (c.g.swap_size > 512) c.g.swap_size = 512;
	c.g.swap_size = (c.g.swap_size + PAGEK - 1) / PAGEK;
	c.g.swap_size *= PAGEK;

#if defined(MSDOS)
#if defined(SYSMGR)
	c.g.special = 'J';
	c.g.screen_type = 'M';
	c.g.screen_size = 'J';
	c.g.key_type = 'J';
#else
	if (c.g.special == 'J') {
		c.g.screen_type = 'M';
		c.g.screen_size = 'J';
		c.g.key_type = 'J';
		}
#endif
#endif

	WFixup(&c.d);
	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Read and return the next character. */

int
Ini_Next()
	{
	if (bufptr >= &buffer[buflen]) {
		buflen = read(fd, buffer, sizeof(buffer));
		if (buflen <= 0) return(EOF);
		bufptr = buffer;
		}
	if (*bufptr == NL) line++;
	return(*bufptr++ & 0xFF);
	}


/* ------------------------------------------------------------ */

/* Read and return one token. */

FLAG
Ini_Token(token, len)
	char *token;
	int len;
	{
	int c;

/* skip whitespace */

	do	{
		c = Ini_Next();
		if (c == '#') {
			do	{
				c = Ini_Next();
				} while (c != EOF && c != NL);
			}
		} while (xisgray(c));

/* copy token */

	while (--len > 0 && !xisgray(c)) {
		*token++ = c;
		c = Ini_Next();
		}
	*token = NUL;

/* check for trailing comment */

	if (c == '#') {
		do	{
			c = Ini_Next();
			} while (c != EOF && c != NL);
		}
	return(c != EOF);
	}


/* end of INI.C -- Routines to Load the Initialization File */
